#include "global.h"

static int test_check P__((int colour, Move moveptr));

/*
 * PARSE_MOVE
 *
 * Parse an algebraic move, check for ambiguities and returns the index of
 * the move in the movelist. 
 *
 * Returns -1 if an error occured.
 */
int
parse_move(token, movetext, halfmove, filename, linenum)
	MoveToken token;
	char *movetext, *filename;
	int halfmove;
	u_long linenum;
{
    int i, flags, colour, movenum;
    u_char piece, prom;
    u_char file_from, rank_from, file_to, rank_to, to;
    Move moveptr;
	char *s;

	s = movetext;
	colour = to_colour(halfmove);
    flags = 0;
    prom = 0;
    file_from = 0xff;
    rank_from = 0xff;

    switch (token) {
    case MT_PAWN_MOVE:
    case MT_PAWN_MOVE_P:
        piece = PAWN;
        file_to = *s++ - 'a';
       	rank_to = *s++ - '1';
        if (token == MT_PAWN_MOVE_P) {
            if (*s == '=')
                s++;
            prom = text2piece(*s++);
            flags |= PROMOTION;
        }
        break;
    case MT_PAWN_CAPTURE:
	case MT_PAWN_CAPTURE_SHORT:
    case MT_PAWN_CAPTURE_P:
	case MT_PAWN_CAPTURE_SHORT_P:
        piece = PAWN;
        file_from = *s++ - 'a';
        if (*s == 'x' || *s == 'X')
            s++;                       /* optional for pawn capture */
        file_to = *s++ - 'a';
		if (token == MT_PAWN_CAPTURE_SHORT_P) {
			if (colour) {
				/* black promoting */
				rank_from = 1;
				rank_to = 0;
			} else {
				/* white promoting */
				rank_from = 6;
				rank_to = 7;
			}
		} else if (token == MT_PAWN_CAPTURE || token == MT_PAWN_CAPTURE_P) {
			rank_to = *s++ - '1';
		}
        flags |= CAPTURE;
        if (token == MT_PAWN_CAPTURE_P || token == MT_PAWN_CAPTURE_SHORT_P) {
            if (*s == '=')
                s++;
            prom = text2piece(*s++);
            flags |= PROMOTION;
        }
        break;
    case MT_PIECE_MOVE:
    case MT_PIECE_MOVE_F:
    case MT_PIECE_MOVE_R:
    case MT_PIECE_MOVE_FR:
        piece = text2piece(*s++);
        if (token == MT_PIECE_MOVE_F || token == MT_PIECE_MOVE_FR) {
            file_from = *s++ - 'a';
        }
        if (token == MT_PIECE_MOVE_R || token == MT_PIECE_MOVE_FR) {
            rank_from = *s++ - '1';
        }
        file_to = *s++ - 'a';
        rank_to = *s++ - '1';
        break;
    case MT_PIECE_CAPTURE:
    case MT_PIECE_CAPTURE_F:
    case MT_PIECE_CAPTURE_R:
    case MT_PIECE_CAPTURE_FR:
        piece = text2piece(*s++);
        if (token == MT_PIECE_CAPTURE_F || token == MT_PIECE_CAPTURE_FR) {
            file_from = *s++ - 'a';
        }
        if (token == MT_PIECE_CAPTURE_R || token == MT_PIECE_CAPTURE_FR) {
            rank_from = *s++ - '1';
        }
        s++;                           /* 'x' or 'X' */
        file_to = *s++ - 'a';
        rank_to = *s++ - '1';
        flags |= CAPTURE;
        break;
    case MT_KSIDE_CASTLE:
        piece = KING;
        if (colour) {                  /* black */
            file_from = 4;
            rank_from = 7;
            file_to = 6;
            rank_to = 7;
        } else {                       /* white */
            file_from = 4;
            rank_from = 0;
            file_to = 6;
            rank_to = 0;
        }
        flags |= SHORT_CASTLE;
        s += 3;
        break;
    case MT_QSIDE_CASTLE:
        piece = KING;
        if (colour) {                  /* black */
            file_from = 4;
            rank_from = 7;
            file_to = 2;
            rank_to = 7;
        } else {                       /* white */
            file_from = 4;
            rank_from = 0;
            file_to = 2;
            rank_to = 0;
        }
        flags |= LONG_CASTLE;
        s += 5;
        break;
    default:
        error("parse_move(): invalid token %d", (int)token);
        return -1;
    }

    if (colour)
        piece |= 8;

    gen_movelist(colour);

	if (token == MT_PAWN_CAPTURE_SHORT) {
		/* try to determine the to rank of the pawn capture */
		for (movenum = -1, i = 0; i < moveidx; i++) {
		    moveptr = &movelist[i];
			if (to_file(moveptr->from) == file_from &&
				to_file(moveptr->to) == file_to &&
				(cb_board[moveptr->from] & 0x7) == PAWN &&
				(cb_board[moveptr->to] & 0x7) == PAWN) {
				rank_to = to_rank(moveptr->to);
				movenum = i;
				break;
			}
		}
	}
	if (movenum == -1) {
        error("%s(%lu): unable to determine rank of capture \"%s%s\";",
        	filename, linenum, format_movenum(halfmove), movetext);
        dump_board(cb_board, stdout);
        return -1;
    }


    to = to_offset(file_to, rank_to);

    for (movenum = -1, i = 0; i < moveidx; i++) {
        moveptr = &movelist[i];

        if (moveptr->to != to)
            continue;
        if ((u_char) cb_board[moveptr->from] != piece)
            continue;
        if (file_from != 0xff &&
          (u_char) to_file(moveptr->from) != file_from)
            continue;
        if (rank_from != 0xff &&
          (u_char) to_rank(moveptr->from) != rank_from)
            continue;
        if (flags & PROMOTION && moveptr->prom != prom)
            continue;
        if (test_check(colour, moveptr))
            continue;
        movenum = i;
        break;
    }

    if (movenum == -1) {
        error("%s(%lu): illegal %s \"%s%s\"", filename, linenum,
          format_movetype(token), format_movenum(halfmove), movetext);
        dump_board(cb_board, stdout);
        return -1;
    }

 /* check if the move is ambiguous by looking for another... */
    if (file_from == 0xff && rank_from == 0xff) {
        for (i = movenum + 1; i < moveidx; i++) {
            moveptr = &movelist[i];

            if (moveptr->to != to)
                continue;
            if ((u_char) cb_board[moveptr->from] != piece)
                continue;
            if (file_from != 0xff &&
              (u_char) to_file(moveptr->from) != file_from)
                continue;
            if (rank_from != 0xff &&
              (u_char) to_rank(moveptr->from) != rank_from)
                continue;
            if (flags & PROMOTION && moveptr->prom != prom)
                continue;
        /* file specified, and this file not the same */
            if (file_from != 0xff &&
              (to_file(moveptr->from) != to_file(movelist[i].from)))
                continue;
        /* rank specified, and this rank not the same */
            if (rank_from != 0xff &&
              (to_rank(moveptr->from) != to_rank(movelist[i].from)))
                continue;
            if (test_check(colour, moveptr))
                continue;
        /* must be ambiguous */
            error("%s(%lu): ambiguous %s \"%s%s\"", filename,
              linenum, format_movetype(token),
              format_movenum(halfmove), movetext);
            dump_board(cb_board, stdout);
            return -1;
        }
    }

    return movenum;
}

/*
 * TEST_CHECK
 *
 * Test if a move is illegal because it results in check.
 */
static int
test_check(colour, moveptr)
    int colour;
    Move moveptr;
{
    u_char saved_board[64];
    int saved_ep, ret;

    colour = colour ? 0 : 8;           /* reverse colour */
    copy_board(cb_board, saved_board);
    saved_ep = cb_enpassant;
    do_move(moveptr->from, moveptr->to, moveptr->prom);
    ret = is_check(colour);
    cb_enpassant = saved_ep;
    copy_board(saved_board, cb_board);
    return ret;
}
