/*
 * Memory Paging System (emulator for non-MSDOS machines)
 */
#include "global.h"

#define PAGE_SZ         32768          /* size of a page */
#define NPAGES          288            /* number of pages */

#define FREE    0                      /* unallocated page */
#define ALLOC   1                      /* allocated page */

static u_short blksize;                /* size of a block */
static int bpp;                        /* blocks per page */
static char **page_index = NULL;       /* page index table */
static int *page_offset = NULL;        /* page offset table */
static char *tmp_block = NULL;         /* used during copy */
static int page, blocknum;             /* page and block calculated by
                                          get_blk() */

static PTR get_blk P__((u_long block));
/*
 * MPS_INIT
 *
 * Allocates memory for the page index and initilises the MemPage system.
 *
 * Returns: 0 upon success, else -1.
 */
int
mps_init(blocksize)
    u_short blocksize;
{
    register int i;
    int offset;

    mps_cleanup();
    blksize = blocksize;

    page_index = (char **) malloc(NPAGES * sizeof(char *));
    if (!page_index)
        goto init_error;

    bpp = PAGE_SZ / blocksize;

    page_offset = (int *) malloc(bpp * sizeof(int));
    if (!page_offset) {
        mps_cleanup();
        goto init_error;
    }
    for (i = offset = 0; i < bpp; i++) {
        page_offset[i] = offset;
        offset += blocksize;
    }

    tmp_block = malloc(blocksize);
    if (!tmp_block) {
        mps_cleanup();
        goto init_error;
    }
    bzero((char *) page_index, NPAGES * sizeof(char *));

    return 0;

init_error:
    error("memory init error");
    return -1;
}

/*
 * MPS_CLEANUP
 *
 * Free memory used by the MemPage System.
 */
int
mps_cleanup()
{
    register int i;

    if (page_index) {
        for (i = 0; i < NPAGES; i++)
            if (page_index[i])
                free(page_index[i]);
        free((char *) page_index);
        page_index = NULL;
    }
    if (page_offset) {
        free((char *) page_offset);
        page_offset = NULL;
    }
    if (tmp_block) {
        free(tmp_block);
        tmp_block = NULL;
    }
    return 0;
}

/*
 * MPS_ADDBLK
 *
 * Add a block, allocating memory pages as required.
 */
PTR
mps_addblk(block)
    u_long block;
{
    PTR ptr;

    ptr = get_blk(block);
    if (ptr)
        return ptr;
    page_index[page] = malloc(PAGE_SZ);
    if (!page_index[page]) {
        error("memory exhausted");
        return NULL;
    }
    return page_index[page] + page_offset[blocknum];
}

/*
 * MPS_GETBLK
 *
 * Get the address of a block, or NULL if not allocated.
 */
PTR
mps_getblk(block)
    u_long block;
{
    PTR ptr;

    ptr = get_blk(block);
    if (ptr)
        return ptr;
    error("memory error");
    return NULL;
}

/*
 * GET_BLK
 *
 * Get the address of a block, or NULL if not allocated.
 */
static PTR
get_blk(block)
    u_long block;
{
    page = block / (u_long) bpp;
    blocknum = block % (u_long) bpp;
    if (page_index[page])
        return page_index[page] + page_offset[blocknum];
    return NULL;
}

/*
 * MPS_SWAPBLK
 *
 * Swap the contents of two blocks.
 */
int
mps_swapblk(block1, block2)
    u_long block1, block2;
{
    PTR ptr1, ptr2;

    ptr1 = mps_getblk(block1);
    if (!ptr1)
        return -1;
    ptr2 = mps_getblk(block2);
    if (!ptr2)
        return -2;

    bcopy((char *) ptr1, tmp_block, blksize);
    bcopy((char *) ptr2, (char *) ptr1, blksize);
    bcopy(tmp_block, (char *) ptr2, blksize);
    return 0;
}

/*
 * MPS_COPYBLK
 *
 * Copy the contents of one block to another.
 */
int
mps_copyblk(block1, block2)
    u_long block1, block2;
{
    PTR ptr1, ptr2;

    ptr1 = mps_getblk(block1);
    if (!ptr1)
        return -1;
    ptr2 = mps_getblk(block2);
    if (!ptr2)
        return -1;

    bcopy((char *) ptr1, (char *) ptr2, blksize);
    return 0;
}

/*
 * MPS_TOBLK
 *
 * Copy data from external memory into a block.
 */
int
mps_toblk(block, mem)
    u_long block;
    PTR mem;
{
    PTR ptr;

    ptr = mps_getblk(block);
    if (!ptr)
        return -1;
    bcopy((char *) mem, (char *) ptr, blksize);
    return 0;
}
/*
 * MPS_FROMBLK
 *
 * Copy data to external memory into a block.
 */
int
mps_fromblk(block, mem)
    u_long block;
    PTR mem;
{
    PTR ptr;

    ptr = mps_getblk(block);
    if (!ptr)
        return -1;
    bcopy((char *) ptr, (char *) mem, blksize);
    return 0;
}
