;***************************************************************************
; Memory Paging System -- Copyright (c)1993 Andy Duplain.
;
; File:		mempage.asm
;
; Description:	Memory Paging System functions.
;
; Change Log
; ----------
; 1.0	22/10/93	Initial.
;
;***************************************************************************

	DOSSEG
	.MODEL	large
	.286

;===========================================================================

PAGE_SZ		EQU	32768		; size of a page
PAGE_ALLOC_SZ	EQU	(PAGE_SZ / 16)	; page size in paragraphs
NPAGES		EQU	288		; number of pages
INDEX_ALLOC_SZ	EQU	((NPAGES * 4) / 16)	; index size (paragraphs)

; identifiers used to define where a page is in memory (page index entries):
;
; ident		value
; -----		-----
; FREE		unused
; CONV		conventional memory segment
; EMS		EMS page handle

FREE	EQU	0		; unallocated page
CONV	EQU	1		; conventional memory
EMS	EQU	2		; expanded memory

;===========================================================================

	.DATA

blksize		dw	?	; size of a block
bpp		dw	?	; number of blocks per page

ems_frame	dw	2 DUP(?) ; EMS pages currently in the page frames
ems_phys	dw	2 DUP(?) ; EMS page frame segments

page_index	dw 	0	; page index segment (holds list of
				; ident-value pairs used to find page)
page_offset	dw	0	; page offset segment (holds list of
				; offsets for each block in a page).

tmp_block	dw	0	; segment of temporary block used during copy

ems_avail	db	0	; EMS-available flag
ems_alloc	db	0	; "allocate from EMS" flag
ems_next	db	0	; next page frame to use when paging in
ems_name	db	'EMMXXXX0', 0	; name of EMS "device"
mem_init_err	db	'memory initialization error', 0
mem_clnup_err	db	'memory cleanup error', 0
mem_err		db	'memory error', 0
mem_ex_err	db	'memory exhausted', 0

	align

;===========================================================================

	.CODE

	PUBLIC	_mps_init, _mps_cleanup, _mps_addblk, _mps_getblk
	PUBLIC	_mps_swapblk, _mps_copyblk, _mps_toblk, _mps_fromblk

	EXTRN	C error:FAR

;---------------------------------------------------------------------------
; _MPS_INIT
;
; int mps_init(u_short blocksize)
;	blocksize:	the size of the blocks held in the pages.
;
; Allocate memory for the page index and initilise the MemPage System.
;
; Returns: 0 upon sucess, else -1.
;---------------------------------------------------------------------------

_mps_init	PROC FAR

	push	bp
	mov	bp, sp
	push	bx
	push	cx
	push	dx
	push	di
	push	es       

	; make sure any previous memory is freed
	call	FAR PTR _mps_cleanup

	; get parameters from stack

	mov	ax, WORD PTR [bp+6]
	inc	ax
	and	ax, 0fffeh		; make blocksize word-aligned
	mov	blksize, ax

	; allocate memory for page index

	mov	ah, 48h		; allocate mem block
	mov	bx, INDEX_ALLOC_SZ	; number of paragraphs to allocate
	int	21h
	jnc	alloc_ok1
	jmp	init_error
alloc_ok1:
	mov	page_index, ax	; save address

	; calculate how many block per page

	xor	dx, dx
	mov	ax, PAGE_SZ	; 32-bit page size
	div	blksize
	mov	bpp, ax		; quotient

	; create page-offset table
	mov	bx, ax
	shr	bx, 3		; (bpp * 2) / 16 == bpp >> 3
	inc	bx		; plus one for luck :-/
	mov	ah, 48h		; allocate mem block
	int	21h
	jnc	po_ok

	call	FAR PTR _mps_cleanup	; free page index table
	jmp	init_error

po_ok:
	mov	page_offset, ax
	mov	es, ax		; start of table
	xor	di, di		; offset into table
	mov	cx, bpp		; number of blocks
	xor	ax, ax		; offset
	cld
pi_loop:
	stosw			; store offset
	add	ax, blksize
	loop	pi_loop

	; allocate memory for temporary copy block

	mov	bx, blksize
	shr	bx, 4		; size in paragraphs
	inc	bx		; plus one for luck :-)
	mov	ah, 48h		; allocate memory block
	int	21h
	jnc	tba_ok

	call	FAR PTR _mps_cleanup
	jmp	SHORT init_error

tba_ok:
	mov	tmp_block, ax

	; clear all page index entries

	mov	es, page_index
	xor	di, di
	mov	cx, NPAGES * 2	; 2 words for each page
	xor	ax, ax
	cld
	rep	stosw

	; check if EMS is available

	mov	dx, OFFSET ems_name ; "EMMXXXX0"
	mov	ax, 3d00h	; open file/device, mode = 00h
	int	21h
	jc	no_ems

	mov	bx, ax		; handle
	mov	ax, 4400h	; get device information
	int	21h
	jc	no_ems
	and	dx, 80h		; bit-7 = 1 if character device
	jz	no_ems

	mov	ax, 4407h	; get output status
	int 	21h
	jc	no_ems
	or	al, al		; test device status
	jz	no_ems

	mov	ah, 3eh		; close file/device
	int	21h
	jc	no_ems

	; get physical segment address of EMS page frame
	mov	ax, 4100h	; get page frame address
	int	67h
	test	ah, 0ffh
	jnz	no_ems
	mov	ems_phys, bx
	add	bx, 800h
	mov	WORD PTR ems_phys+2, bx

	mov	ems_avail, 1

	; clear EMS variables

	mov	ems_frame, 0
	mov	WORD PTR ems_frame+2, 0
	mov	ems_alloc, 0
	jmp	SHORT init_ok

no_ems:
	mov	ems_avail, 0

init_ok:
	xor	ax, ax
	jmp	SHORT init_exit

init_error:
	mov	ax, OFFSET mem_init_err
	call	call_error

init_exit:
	pop	es
	pop	di
	pop	dx
	pop	cx
	pop	bx
	pop	bp
	ret

_mps_init	ENDP

;---------------------------------------------------------------------------
; _MPS_CLEANUP
;
; int mps_cleanup(void)
;
; free memory used by the MemPage System.
;---------------------------------------------------------------------------

_mps_cleanup	PROC FAR

	push	bx
	push	cx
	push	dx
	push	es

	test	page_index, 0ffffh
	jz	no_page_index	; not set

	; free each page in the index

	mov	es, page_index
	mov	cx, NPAGES	; number of pages to free
	xor	bx, bx		; offset into page index
free_loop:
	mov	ax, WORD PTR es:[bx]
	mov	dx, WORD PTR es:[bx+2]
	mov	WORD PTR es:[bx], 0  	; clear entries in table
	mov	WORD PTR es:[bx+2], 0
	cmp	ah, CONV
	jnz	not_conv

	; free conventional memory block

	push	es
	mov	es, dx
	mov	ah, 49h		; release memory block
	int	21h
	pop	es
      	jc	cleanup_error

	jmp	SHORT next_free

not_conv:
	cmp	ah, EMS
	jnz	next_free

	; free EMS handle
	mov	ah, 45h		; release handle and expanded memory
	int	67h
      	test	ah, 0ffh
      	jnz	cleanup_error
	
next_free:
	add	bx, 4
	loop	free_loop

	; free the page index

	mov	es, page_index
	mov	page_index, 0
	mov	ah, 49h		; free mem block
	int 	21h
      	jc	cleanup_error

	; free page offset table
no_page_index:
	test	page_offset, 0ffffh
	jz	no_page_offset

	mov	es, page_offset
	mov	page_offset, 0
	mov	ah, 49h		; free mem block
	int	21h
      	jc	cleanup_error

no_page_offset:
	test	tmp_block, 0ffffh
	jz	cleanup_ok

	mov	es, tmp_block
	mov	tmp_block, 0
	mov	ah, 49h		; free mem block
	int	21h
	jc	cleanup_error

cleanup_ok:
	xor	ax, ax
	jmp	SHORT cleanup_exit

cleanup_error:
	mov	ax, OFFSET mem_clnup_err
	call	call_error

cleanup_exit:
	pop	es
	pop	dx
	pop	cx
	pop	bx
	ret

_mps_cleanup	ENDP

;---------------------------------------------------------------------------
; _MPS_ADDBLK
;
; PTR mps_addblk(u_long block)
;
; Add a block, allocating memory pages and swapping-in EMS pages as
; required.
;---------------------------------------------------------------------------

_mps_addblk	PROC FAR

	push	bp
	mov	bp, sp
	sub	sp, 2
	push	bx
	push	cx
	push	es

	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jnc	addblk_exit

	mov     WORD PTR [bp-2], ax	; save page offset value

	; allocate a conventional/EMS memory block for the new page

	test	ems_alloc, 0ffh
	jnz	do_ems_alloc

	; conventional memory...

	push	bx
	mov	ah, 48h		; allocate memory block
	mov	bx, PAGE_ALLOC_SZ
	int	21h
	pop	bx
	jc	no_more_conv
	mov	WORD PTR es:[bx+2], ax	; save in page index table
	mov	dx, ax 		; segment
	mov	ah, CONV
	mov	al, 0
	mov	WORD PTR es:[bx], ax	; save attribute
	mov	ax, WORD PTR [bp-2]	; segment offset

	jmp	SHORT addblk_exit

no_more_conv:

	; check if EMS available...

	test	ems_avail, 0ffh
	jz	addblk_error
	mov	ems_alloc, 1	; flag that we are now allocating from EMS

do_ems_alloc:
	push	bx
	mov	ah, 43h		; allocate handle and pages
	mov	bx, 2		; two pages per handle
	int	67h
	pop	bx
	test	ah, 0ffh 	; OK ?
	jnz	addblk_error
	mov	es:[bx+2], dx	; save in page index table
	mov	ah, EMS
	mov	al, 0
	mov	es:[bx], ax	; save attribute

	mov	ax, WORD PTR [bp-2]	; segment offset
	call	ems_pagein	; ensure EMS page is in the page frame
	jnc	addblk_exit

addblk_error:
	mov	ax, OFFSET mem_ex_err
	call	call_error
	xor	ax, ax
	xor	dx, dx

addblk_exit:
	pop	es
	pop	cx
	pop	bx
	mov	sp, bp
	pop	bp
	ret

_mps_addblk	ENDP

;---------------------------------------------------------------------------
; _MPS_GETBLK
;
; PTR mps_getblk(u_long block)
;
; Returns the physical address of a block, else NULL if page in which
; block should exist hasn't been allocated
;---------------------------------------------------------------------------

_mps_getblk	PROC FAR

	push	bp
	mov	bp, sp
	push	bx
	push	es
	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jnc	exit
	xor	ax, ax
	xor	dx, dx
exit:
	pop	es
	pop	bx
	pop	bp
	ret

_mps_getblk	ENDP

;---------------------------------------------------------------------------
; _MPS_SWAPBLK
;
; int mps_swapblk(u_long block1, u_long block2)
;
; Swap the contents of two blocks.
;---------------------------------------------------------------------------

; local variable offsets into stack
BLK1_SEG 	EQU	WORD PTR [bp-8]
BLK1_OFF 	EQU  	WORD PTR [bp-6]
BLK2_SEG	EQU	WORD PTR [bp-4]
BLK2_OFF	EQU	WORD PTR [bp-2]

_mps_swapblk	PROC FAR

	push	bp
	mov	bp, sp
	sub	sp, 8
	push	bx
	push	cx
	push	dx
	push	es

	; get block 1 address
	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jc	swapblk_error
	mov	BLK1_SEG, dx
	mov	BLK1_OFF, ax

	; get get block 2 address
	mov	ax, WORD PTR [bp+10]
	mov	dx, WORD PTR [bp+12]
	call	get_blk
	jc	swapblk_error
	mov	BLK2_SEG, dx
	mov	BLK2_OFF, ax

	; copy from block 1 -> temp

	mov	ax, BLK1_SEG
	mov	bx, BLK1_OFF
	mov	cx, tmp_block
	xor	dx, dx
	call	copy_mem

	; copy from block 2 -> block 1

	mov	ax, BLK2_SEG
	mov	bx, BLK2_OFF
	mov	cx, BLK1_SEG
	mov	dx, BLK1_OFF
	call	copy_mem

	; copy from temp -> block 2

	mov	ax, tmp_block
	xor	bx, bx
	mov	cx, BLK2_SEG
	mov	dx, BLK2_OFF
	call	copy_mem

	xor	ax, ax
	jmp	SHORT swapblk_exit

swapblk_error:
	mov	ax, OFFSET mem_err
	call	call_error

swapblk_exit:
	pop	es
	pop	dx
	pop	cx
	pop	bx
	mov	sp, bp
	pop	bp
	ret

_mps_swapblk	ENDP

;---------------------------------------------------------------------------
; _MPS_COPYBLK
;
; int mps_swapblk(u_long block1, u_long block2)
;
; Copy the contents of block1 to block2.
;---------------------------------------------------------------------------

_mps_copyblk	PROC FAR

	push	bp
	mov	bp, sp
	sub	sp, 4
	push	bx
	push	cx
	push	dx
	push	es

	; get block 1 address
	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jc	copyblk_error
	mov	BLK1_SEG, dx
	mov	BLK1_OFF, ax

	; get get block 2 address
	mov	ax, WORD PTR [bp+10]
	mov	dx, WORD PTR [bp+12]
	call	get_blk
	jc	copyblk_error
	mov	cx, dx		; block2 segment
	mov	dx, ax		; block2 offset
	mov	ax, BLK1_SEG
	mov	bx, BLK1_OFF
	call	copy_mem

	xor	ax, ax
	jmp	SHORT copyblk_exit

copyblk_error:
	mov	ax, OFFSET mem_err
	call	call_error

copyblk_exit:
	pop	es
	pop	dx
	pop	cx
	pop	bx
	mov	sp, bp
	pop	bp
	ret

_mps_copyblk	ENDP

;---------------------------------------------------------------------------
; _MPS_TOBLK
;
; int mps_toblk(u_long block, PTR block)
;
; Copy data from external memory into a block.
;---------------------------------------------------------------------------

_mps_toblk	PROC FAR

	push	bp
	mov	bp, sp
	push	bx
	push	cx
	push	dx
	push	es

	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jc	toblk_error
	mov	cx, dx		; destination segment
	mov	dx, ax		; destination offset
	mov	ax, WORD PTR [bp+12]	; source segment
	mov	bx, WORD PTR [bp+10]	; source offset
	call	copy_mem

	xor	ax, ax
	jmp	SHORT toblk_exit

toblk_error:
	mov	ax, OFFSET mem_err
	call	call_error

toblk_exit:
	pop	es
	pop	dx
	pop	cx
	pop	bx
	pop	bp
	ret

_mps_toblk	ENDP

;---------------------------------------------------------------------------
; _MPS_FROMBLK
;
; int mps_fromblk(u_long block, PTR block)
;
; Copy data from a block into external memory.
;---------------------------------------------------------------------------

_mps_fromblk	PROC FAR

	push	bp
	mov	bp, sp
	push	bx
	push	cx
	push	dx
	push	es

	mov	ax, WORD PTR [bp+6]
	mov	dx, WORD PTR [bp+8]
	call	get_blk
	jc	fromblk_error
	mov	bx, ax		; source offset
	mov	ax, dx		; source segment
	mov	cx, WORD PTR [bp+12]	; destination segment
	mov	dx, WORD PTR [bp+10]	; destination offset
	call	copy_mem

	xor	ax, ax
	jmp	SHORT fromblk_exit

fromblk_error:
	mov	ax, OFFSET mem_err
	call	call_error

fromblk_exit:
	pop	es
	pop	dx
	pop	cx
	pop	bx
	pop	bp
	ret

_mps_fromblk	ENDP

;---------------------------------------------------------------------------
; GET_BLK
;
; Get physical address of a block.
;
; Entry: DX:AX = block number
; Exit:  DX = segment (page frame if EMS block)
;        AX = offset
;	 BX = offset into page index table for block's page
;	 ES = page index table segment
;	 Carry flag set if page not allocated or out of range.
;---------------------------------------------------------------------------

get_blk		PROC NEAR

	push	cx

	; find the page/offset

	test	dx, 8000h	; sign bit set ?
	jnz	getblk_error

	div	bpp		; block / bpp = page/block
	mov	bx, dx
	mov	dx, ax		; page number
	shl	bx, 1		; * 2 for offset into page offset table
	mov	es, page_offset
	mov	cx, WORD PTR es:[bx] ; block offset
	; CX = offset, DX = page number

	cmp	dx, NPAGES
	jae	getblk_error
	
	mov	es, page_index
	mov	bx, dx
	shl	bx, 2		; to double-word index
	mov	dx, WORD PTR es:[bx]	; get attribute word
	cmp	dh, CONV
	jnz	not_conv1
	mov	dx, WORD PTR es:[bx+2]	; segment
	jmp	SHORT getblk_exit
not_conv1:
	cmp	dh, EMS
	jnz	getblk_error
	mov	dx, WORD PTR es:[bx+2]	; EMS handle
	call	ems_pagein
	jc	getblk_error

getblk_ok:
	lahf
	and	ah, 0feh	; clear carry flag
	sahf
	jmp	SHORT getblk_exit

getblk_error:
	lahf
	or	ah, 1		; set carry flag
	sahf

getblk_exit:
	mov	ax, cx		; block offset
	pop	cx
	ret

get_blk		ENDP

;---------------------------------------------------------------------------
; EMS_PAGEIN
;
; Ensure that the EMS page with the supplied handle is swapped into memory.
;
; Entry: DX = EMS page handle
; Exit:  DX = Segment of physical page
;---------------------------------------------------------------------------

ems_pagein	PROC NEAR

	push	ax
	push	bx

	; check if already paged-in

	mov	bx, 0
	cmp	dx, ems_frame
	jz	epi_in
	inc	bx
	inc	bx
	cmp	dx, WORD PTR ems_frame+2
	jz	epi_in

	; page-in the ems pages (0 and 1)

	mov	ah, 44h		; map expanded memory page
	mov	al, ems_next
	xor	bx, bx
	int	67h
	test	ah, 0ffh
	jnz	epi_error

	mov	ah, 44h	 	; map expanded memory page
	mov	al, ems_next
	inc	al
	mov	bx, 1
	int	67h
	test	ah, 0ffh
	jnz	epi_error

	mov	al, ems_next
	xor	bx, bx
	mov	bl, al		; frame we chose for mapping
	mov	WORD PTR ems_frame[bx], dx	; save EMS handle

	xor	al, 2		; 0 -> 2, 2 -> 0
	mov	ems_next, al

epi_in:
	mov	dx, WORD PTR ems_phys[bx]
	lahf
	and	ah, 0feh	; clear carry flag	
	sahf
	jmp	SHORT epi_exit

epi_error:
	lahf
	or	ah, 1		; set carry flag
	sahf

epi_exit:
	pop	bx
	pop	ax
	ret

ems_pagein	ENDP

;---------------------------------------------------------------------------
; COPY_MEM
;
; Copy a blocksize of memory.
;
; Entry: AX:BX = Source address
;	 CX:DX = Destination address
;---------------------------------------------------------------------------

copy_mem	PROC NEAR

	push	si
	push	di
	push	ds
	push	es

	mov	es, cx	   	; destination segment
	mov	di, dx	       	; destination offset
	mov	cx, blksize	; size
	shr	cx, 1		; convert to words
	mov	ds, ax		; source segment
	mov	si, bx		; source offset
	cld
	rep	movsw

	pop	es
	pop	ds
	pop	di
	pop	si
	ret
copy_mem	ENDP

;---------------------------------------------------------------------------
; CALL_ERROR
;
; Call the C-function "error" with the given message.
;
; Entry: AX = offset of message in data segment
; Exit:  AX = -1 (for returning).
;---------------------------------------------------------------------------

call_error	PROC NEAR

	push	ds
	push	ax
	call	error
	add	sp, 4
	mov	ax, -1
	ret

call_error	ENDP
	END
