#include "global.h"
#ifdef MSDOS
#include <dos.h>
#include <errno.h>

static char *path = NULL, *dbspec = NULL, *retfile = NULL;
static int nfiles = -1;

static struct _find_t db_file;

#define DIRSPEC "\\*.cbf"              /* filespec for directories */

/*
 * FIND_DATABASE_INIT
 *
 * Initialise the find_database() variables etc.
 */
int
find_database_init(filespec)
    char *filespec;
{
    int len, is_a_dir;
    char *cptr;

    find_database_exit();

    is_a_dir = isdir(filespec) == 1 ? 1 : 0;

 /* create the path string containing just the pathname */
    if (is_a_dir) {
        len = strlen(filespec);
    } else {
        cptr = rindex(filespec, PATHSEP);
        if (cptr)
            len = cptr - filespec;
        else
            len = 0;
    }
    if (len) {
        path = mem_alloc(len + 1);
        if (!path)
            return -1;
        strncpy(path, filespec, len);
    } else {
        path = NULL;
    }

    if (is_a_dir) {
        dbspec = mem_alloc(len + sizeof(DIRSPEC) + 1);
        if (!dbspec) {
            find_database_exit();
            return -1;
        }
        strcpy(dbspec, filespec);
        strcat(dbspec, DIRSPEC);
    } else {
        kill_ext(filespec);            /* damages input file! */
        dbspec = derive_name(filespec, ".cbf");
        if (!dbspec) {
            find_database_exit();
            return -1;
        }
    }

 /* create buffer for output filename */
    retfile = mem_alloc(len + MAX_FNAMELEN + 2);
    if (!retfile) {
        find_database_exit();
        return -1;
    }
    nfiles = 0;
}

/*
 * FIND_DATABASE
 *
 * Find the next (or first) database that matches the filespec passed to
 * find_database_init().
 */
char *
find_database()
{
    int status;

    if (nfiles == -1)
        return NULL;

    if (nfiles == 0) {
        status = _dos_findfirst(dbspec, _A_NORMAL, &db_file);
        if (status)
            error("%s: %s", dbspec, strerror(errno));
    } else {
        status = _dos_findnext(&db_file);
    }

    nfiles++;

    if (status == 0) {
        if (path) {
            strcpy(retfile, path);
            strcat(retfile, "\\");
        } else {
            *retfile = '\0';
        }
        strcat(retfile, db_file.name);
        strlower(retfile);
        kill_ext(retfile);
        return retfile;
    }
    return NULL;
}

/*
 * FIND_DATABASE_EXIT
 *
 * Clearup
 */
int
find_database_exit()
{
    if (path) {
        free(path);
        path = NULL;
    }
    if (dbspec) {
        free(dbspec);
        dbspec = NULL;
    }
    if (retfile) {
        free(retfile);
        retfile = NULL;
    }
    nfiles = -1;

    return 0;
}

#else                                  /* !MSDOS */

static char *filename;
/*
 * FIND_DATABASE_INIT
 *
 * Initialise the find_database() variables etc.
 */
int
find_database_init(filespec)
    char *filespec;
{
    int len;

    /* if the filespec has a period in it, then check if the name ends with
       '.cbf' */
    len = strlen(filespec);
    if (index(filespec, '.')) {
        if (len > 4 && strcmp(filespec + (len - 4), ".cbf") == 0)
            filename = filespec;
        else
            filename = NULL;
    } else {
        /* assume OK */
        filename = filespec;
    }
    return 0;
}

/*
 * FIND_DATABASE
 *
 * Find the next (or first) database that matches the filespec passed to
 * find_database_init().
 */
char *
find_database()
{
    char *ret;

    if (filename) {
        ret = filename;
        filename = NULL;
        return ret;
    }
    return NULL;;
}

/*
 * FIND_DATABASE_EXIT
 *
 * Clearup
 */
int
find_database_exit()
{
    filename = NULL;
    return 0;
}

#endif                                 /* MSDOS */
