/***************************************************************************
 * KTESTS.C -- test functions for .ks files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          12/11/93        Initial.
 * 1.1          28/01/94        Added square testing to test_material().
 *                              Added test_xxx_p() functions.
 ***************************************************************************/

#include "global.h"
#include "kcompile.h"
#include "regexp.h"

#ifdef DEBUG
#define KTESTS_DEBUG
#endif

extern u_long first, last;             /* in cbkey.c */
extern Database db;                    /* in cbkey.c */
extern char *kfname;                   /* in cbkey.c */
extern File keyfile;                   /* in cbkey.c */
extern int quiet;                      /* in cbkey.c */

/*
 * structure to hold board position and move data
 */
struct testmove {
    u_char board[64];                  /* position of board prior to move */
    short halfmove;                    /* halfmove number */
    u_char file_from, rank_from;       /* piece moved from this square... */
    u_char file_to, rank_to;           /* to this square */
    u_short flags;                     /* flags generated during move */
    u_char enpassant;                  /* enpassant mask for this move */
    u_char prom;                       /* promoted piece */
    u_char check_tested;               /* CHECK_TESTED and/or MATE_TESTED */
};
typedef struct testmove *TestMove;

#define NONE_TESTED     0
#define CHECK_TESTED    1
#define MATE_TESTED     2

#define MAX_TESTMOVES 256              /* maximum number of supported
                                          halfmoves */
static TestMove testmoves;
static int last_testmove;              /* first and last halfmove number */
static int movenum;                    /* move number currently being tested */
static int abort_test;                 /* flag used by hmovenum test */

static u_long blocknum;                /* MPS block number to assign/allocate */
static struct game game;               /* game being classified */
static struct key masterkey;           /* dummy key to represent master
                                          record */
static u_char *board;                  /* pointer to board being tested */

static int wsqs[32] = {
    1, 3, 5, 7, 8, 10, 12, 14, 17, 19, 21, 23, 24, 26, 28, 30,
    33, 35, 37, 39, 40, 42, 44, 46, 49, 51, 53, 55, 56, 58, 60, 62
};
static int bsqs[32] = {
    0, 2, 4, 6, 9, 11, 13, 15, 16, 18, 20, 22, 25, 27, 29, 31,
    32, 34, 36, 38, 41, 43, 45, 47, 48, 50, 52, 54, 57, 59, 61, 63
};

static int set_testmoves P__((int halfmove));
static int add_key_block P__((Key key, Key parentkey));
static int test_main_key P__((Key key));
static int test_key P__((Key key));
static int run_test P__((KStest test));
static int test_board P__((u_char * test_board));
static int test_material P__((PTest ptest));
static int test_movement P__((PTest ptest));
static int test_capture P__((PTest ptest));
static int test_pass_p P__((PTest ptest));
static int test_utd_p P__((PTest ptest));
static int test_iso_p P__((PTest ptest));
static int test_dbl_p P__((PTest ptest));
static int test_dbliso_p P__((PTest ptest));
static int test_value P__((PTest ptest, int value));
static int test_check P__((int mate));
static int remove_empties P__((void));
static int tidy_index P__((void));
static int compress P__((void));
static int add_to_kindex P__((Key key, u_long entry));
static KR allocblk P__((void));

#define TITLE "[ Created using CBKEY (c)1993-94 Andy Duplain ]"

/*
 * CLASSIFY_GAMES
 *
 * Classify the games in a database.
 */
int
classify_games()
{
    KR master, index;
    u_long i, ngames, count;
    int progress, last_report, j;
    Key testkey;

 /* allocate enough memory to store the maximum number of games */
    testmoves = (TestMove) mem_alloc(sizeof(struct testmove) *
      (MAX_TESTMOVES + 1));
    if (!testmoves)
        return -1;

    blocknum = 0L;

 /* build the key-record structure in memory */
    if (mps_init(sizeof(struct kr)) < 0) {
        free(testmoves);
        return -1;
    }
 /* allocate the master and index records */
    master = allocblk();
    if (!master)
        goto fail;
    master->type = KR_MASTER_TYPE;
    master->next = 1L;
    strcpy((char *) master->data.m.title, TITLE);
    index = allocblk();
    if (!index)
        goto fail;
    index->type = KR_INDEX_TYPE;
    masterkey.block = 0L;
    masterkey.iblock = 1L;

 /* build the key-tree in memory */
    if (add_key_block(rootkey, &masterkey) < 0)
        goto fail;

    ngames = (last - first) + 1;
    last_report = -1;
    bzero((char *) &game, sizeof(struct game));

    for (i = first; i <= last; i++) {
        last_testmove = 0;
        game_tidy(&game);
        if (info_tests_only) {         /* don't need move info etc. */
            game.num = i;
            if (read_info(db, &game) < 0)
                continue;
            if (is_deleted(game.header))
                continue;
            last_testmove = 1;
        } else {
            game.num = i;
            if (read_game(db, &game) < 0)
                continue;
            if (is_deleted(game.header))
                continue;
            copy_board(game.board, testmoves[0].board);
            testmoves[0].enpassant = game.ep;
            process_moves(&game, set_testmoves);
        }

    /* run the test for all the main keys */
        for (testkey = rootkey; testkey; testkey = testkey->siblist)
            test_main_key(testkey);

        if (!quiet) {
            progress = (int) ((i * 100L) / (ngames));
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
    }

 /* remove empty keys */
    if (remove_empty_keys) {
        if (!quiet)
            output("removing empty keys ");
        do {
            j = remove_empties();
            if (j < 0) {
                error("\nerror removing empty keys!");
                goto fail;
            }
        } while (j);
        if (!quiet)
            newline();
    }
 /* compress keyfile */
    if (compress_keyfile) {
        if (!quiet)
            output("compressing keyfile ");
        if (compress() < 0) {
            error("\nerror compressing keyfile!");
            goto fail;
        }
    }
 /* complete the master record */
    master = (KR) mps_getblk(0L);
    if (!master)
        goto fail;

 /* count the number of records in the file */
    if (compress_keyfile) {
        for (i = 0L, count = 0L; i < blocknum; i++) {
            index = (KR) mps_getblk(i);
            if (!index)
                goto fail;
            if (index->type != KR_EMPTY_TYPE)
                count++;
        }
    } else {
        count = blocknum;
    }
    master->data.m.nrecs = count;

 /* count the number of keys in the file */
    for (i = 2L, count = 1L; i < blocknum; i++) {
        index = (KR) mps_getblk(i);
        if (!index)
            goto fail;
        if (index->type == KR_KEY_TYPE)
            count++;
    }
    master->data.m.nkeys = count;
    master->data.m.lastgame = db->ngames;

 /* write keyfile */
    if (!quiet)
        output("writing %s\n", kfname);
    file_seek(keyfile, 0L);
    for (i = 0L; i < blocknum; i++) {
        if (compress_keyfile) {
        /* check it's not an empty key, after compressing */
            master = (KR) mps_getblk(i);
            if (!master)
                goto fail;
            if (master->type == KR_EMPTY_TYPE)
                continue;
        }
        if (kr_write(keyfile, i) < 0)
            goto fail;
    }

    mps_cleanup();
    free(testmoves);
    return 0;

fail:
    mps_cleanup();
    free(testmoves);
    return -1;
}

/*
 * SET_TESTMOVES
 *
 * Function called by process_moves() to setup boards in memory
 */
static int
set_testmoves(halfmove)
    int halfmove;
{
    TestMove moveptr, nmoveptr;

 /* not interested in variations */
    if (halfmove < 0 || var_level)
        return 0;

    if (last_testmove == MAX_TESTMOVES)
        return 0;

    moveptr = &testmoves[last_testmove];
    moveptr->halfmove = halfmove;
    moveptr->prom = lastmove.prom;
    moveptr->file_from = to_file(lastmove.from);
    moveptr->rank_from = to_rank(lastmove.from);
    moveptr->file_to = to_file(lastmove.to);
    moveptr->rank_to = to_rank(lastmove.to);
    moveptr->flags = lastmove_flags;
    moveptr->check_tested = NONE_TESTED;
    nmoveptr = moveptr + 1;
    nmoveptr->enpassant = cb_enpassant;
    copy_board(cb_board, nmoveptr->board);

#if 0
    printf("halfmove=%d flags=%04x move=%c%c%c%c%c\n",
      moveptr->halfmove, moveptr->flags,
      moveptr->file_from + 'a', moveptr->rank_from + '1',
      moveptr->flags & CAPTURE ? 'x' : '-',
      moveptr->file_to + 'a', moveptr->rank_to + '1');
    dump_board(moveptr->board, stdout);
    puts("---------------------------------------");
#endif

    last_testmove++;
    return 0;
}


/*
 * ADD_KEY_BLOCK
 *
 * Allocate and assign a key record block to a key
 *
 * Uses the global variable "blocknum" for the block number to assign.
 */
static int
add_key_block(key, parentkey)
    Key key, parentkey;
{
    KR kr_addr;
    u_long block;

    if (!key)
        return 0;

 /* build key record */
    block = blocknum;
    kr_addr = allocblk();
    if (!kr_addr)
        return -1;
    kr_addr->type = KR_KEY_TYPE;
    kr_addr->prev = parentkey->block;
    strcpy((char *) kr_addr->data.k.name, (char *) key->name);

 /* record key index block in key */
    key->block = block;

 /* add key record number to parent index */
    if (add_to_kindex(parentkey, block | SUBKEY_MASK) < 0)
        return -1;

    if (key->childlist)
        if (add_key_block(key->childlist, key) < 0)
            return -1;

    if (key->siblist)
        if (add_key_block(key->siblist, parentkey) < 0)
            return -1;

    return 0;
}

/*
 * TEST_MAIN_KEY
 *
 * Do the tests for a main key.
 */
static int
test_main_key(key)
    Key key;
{
    int res, i;

    if (!key) {
        error("test_main_key(NULL) !!!");
        return 0;
    }
    res = 0;

    if (key->childlist) {
        res = test_key(key->childlist);
        if (res && subkey_related)
            return res;
    }
    if (key->tests) {
        for (i = 0, abort_test = 0; i < last_testmove && !abort_test;
          i++) {
            movenum = i;
            if (run_test(key->tests)) {
                if (add_to_kindex(key, game.num) < 0)
                    return -1;
                break;
            }
        }
    }
    return res;
}


/*
 * TEST_KEY
 *
 * Do the tests for each key.
 */
static int
test_key(key)
    Key key;
{
    int res, i;

    if (!key) {
        error("test_key(NULL) !!!");
        return 0;
    }
    res = 0;

    if (key->childlist) {
        res = test_key(key->childlist);
        if (res && subkey_related)
            return res;
    }
    if (key->siblist) {
        res = test_key(key->siblist);
        if (res && subkey_related)
            return res;
    }
    if (key->tests) {
        for (i = 0, abort_test = 0; i < last_testmove && !abort_test;
          i++) {
            movenum = i;
            if (run_test(key->tests)) {
                res = 1;
                if (add_to_kindex(key, game.num) < 0)
                    return -1;
                break;
            }
        }
    }
    return res;
}

/*
 * RUN_TEST
 *
 * Run the test
 */
static int
run_test(test)
    KStest test;
{
    int type, res;

    if (!test)
        return 0;

    if (movenum >= last_testmove)
        return 0;

    type = test->type;

    if (is_conn(type)) {
    /* connector */
        res = run_test(test->left);
        if (!res && type != CONN_OR)
            return 0;                  /* 0 and not an "AND" type test */
        if (res && type == CONN_OR)
            return 1;                  /* 1 and an "OR" type test */

        if (type == CONN_LATER) {
            for (; movenum < last_testmove; movenum++)
                if (run_test(test->right))
                    return 1;
        } else {
            if (type == CONN_THEN)
                movenum++;
            if (run_test(test->right))
                return 1;
        }

        return 0;

    } else {
    /* test */
        switch (type) {
        case TEST_BOARD:
            res = test_board((u_char *) test->ptr);
            break;
        case TEST_MATERIAL:
            res = test_material((PTest) test->ptr);
            break;
        case TEST_MOVEMENT:
            res = test_movement((PTest) test->ptr);
            break;
        case TEST_CAPTURE:
            res = test_capture((PTest) test->ptr);
            break;
        case TEST_HMOVENUM:
            res = ((long) test->ptr == testmoves[movenum].halfmove);
            if (testmoves[movenum].halfmove >= (long) test->ptr)
                abort_test = 1;
            break;
        case TEST_PLAYER:
            res = regexec((regexp *) test->ptr, (char *) game.pinfo);
            break;
        case TEST_SOURCE:
            res = regexec((regexp *) test->ptr, (char *) game.sinfo);
            break;
        case TEST_YEAR:
            res = test_value((PTest) test->ptr, game.year);
            break;
        case TEST_ECO:
            if (game.eco_letter)
                res = test_value((PTest) test->ptr,
                  (game.eco_letter - 'A') * 100 + game.eco_main);
            else
                res = 0;
            break;
        case TEST_RESULT:
            res = test_value((PTest) test->ptr, (int) get_result(game.header));
            break;
        case TEST_NMOVES:
            res = test_value((PTest) test->ptr, game.nmoves);
            break;
        case TEST_PASS_P:
            res = test_pass_p((PTest) test->ptr);
            break;
        case TEST_UTD_P:
            res = test_utd_p((PTest) test->ptr);
            break;
        case TEST_ISO_P:
            res = test_iso_p((PTest) test->ptr);
            break;
        case TEST_DBL_P:
            res = test_dbl_p((PTest) test->ptr);
            break;
        case TEST_DBLISO_P:
            res = test_dbliso_p((PTest) test->ptr);
            break;
        default:
            res = 0;
            break;
        }

        if (res) {
            if (test->negative)
                return 0;
            else
                return 1;
        } else {
            if (test->negative)
                return 1;
            else
                return 0;
        }
    }
}

/*
 * TEST_BOARD
 *
 * Test a specific board set-up
 */
static int
test_board(test_board)
    u_char *test_board;
{
    int i;

    board = testmoves[movenum].board;

    for (i = 0; i < 64; i++, test_board++, board++) {
        if (*test_board == 0xff)
            continue;
        if (*test_board != *board)
            return 0;
    }
    return 1;
}

/*
 * TEST_MATERIAL
 *
 * Test material on the board (before a move was made).
 */
static int
test_material(ptest)
    register PTest ptest;
{
    int wpiece[8], bpiece[8];          /* [0] unused */
    int i, j, piece;
    int start_file, end_file, start_rank, end_rank;
    int *sq_array;

    board = testmoves[movenum].board;

    for (i = 1; i < 7; i++) {
        wpiece[i] = 0;
        bpiece[i] = 0;
    }


    if (ptest->flags & (WHITESQUARE | BLACKSQUARE)) {
    /* square colour test */
        if (ptest->flags & WHITESQUARE)
            sq_array = wsqs;
        else
            sq_array = bsqs;
        for (j = 0; j < 32; j++) {
            i = sq_array[j];
            piece = board[i];
            if (piece & 8)
                bpiece[piece & 7]++;
            else
                wpiece[piece]++;
        }
    } else {
    /* non-square test */

        start_file = 0;
        end_file = 7;
        start_rank = 0;
        end_rank = 7;

        if (ptest->flags & QUEENSIDE)
            end_file = 3;
        else if (ptest->flags & KINGSIDE)
            start_file = 4;
        else if (ptest->flags & WHITESIDE)
            end_rank = 3;
        else if (ptest->flags & BLACKSIDE)
            start_rank = 4;
        if (ptest->file_from != UNSPEC)
            start_file = end_file = ptest->file_from;
        if (ptest->rank_from != UNSPEC)
            start_rank = end_rank = ptest->rank_from;

        for (i = start_file; i <= end_file; i++) {
            for (j = start_rank; j <= end_rank; j++) {
                piece = board[to_offset(i, j)];
                if (piece & 8)
                    bpiece[piece & 7]++;
                else
                    wpiece[piece]++;
            }
        }
    }

 /* if no relational operator specified, this is the same as "== 1" */
    if (ptest->relop == RELOP_NONE) {
        ptest->relop = RELOP_EQ;
        ptest->val = 1;
    }
    i = 0;
    if (ptest->piece == UNSPEC) {
    /* any piece */
        if (ptest->colour == 0) {
        /* number of white pieces */
            for (j = 1; j < 7; j++)
                i += wpiece[j];
        } else {
        /* number of black pieces */
            for (j = 1; j < 7; j++)
                i += bpiece[j];
        }
    } else {
    /* a specific piece */
        if (ptest->colour == 0) {
        /* white piece */
            i = wpiece[ptest->piece];
        } else if (ptest->colour == 1) {
        /* black piece */
            i = bpiece[ptest->piece];
        } else {
        /* any coloured piece */
            i = wpiece[ptest->piece] + bpiece[ptest->piece];
        }
    }

    return test_value(ptest, i);
}

/*
 * TEST_MOVEMENT
 *
 * Test a move
 */
static int
test_movement(ptest)
    register PTest ptest;
{
    register TestMove moveptr;
    int offset;

    moveptr = &testmoves[movenum];

    if (moveptr->flags & CAPTURE)
        return 0;

 /* test colour of moving piece */
    if (ptest->colour != UNSPEC) {
        if ((u_char) to_colour(moveptr->halfmove) != ptest->colour)
            return 0;
    }
 /* test moving piece */
    if (ptest->piece != UNSPEC) {
        offset = to_offset(moveptr->file_from, moveptr->rank_from);
        if (ptest->piece != (u_char) (moveptr->board[offset] & 7))
            return 0;
    }
 /* test square/file/rank moved from */
    if (ptest->file_from != UNSPEC) {
        if (ptest->file_from != moveptr->file_from)
            return 0;
    }
    if (ptest->rank_from != UNSPEC) {
        if (ptest->rank_from != moveptr->rank_from)
            return 0;
    }
 /* test square/file/rank moved to */
    if (ptest->file_to != UNSPEC) {
        if (ptest->file_to != moveptr->file_to)
            return 0;
    }
    if (ptest->rank_to != UNSPEC) {
        if (ptest->rank_to != moveptr->rank_to)
            return 0;
    }
 /* test castling */
    if (ptest->flags & SHORT_CASTLE)
        if (!(moveptr->flags & SHORT_CASTLE))
            return 0;
    if (ptest->flags & LONG_CASTLE)
        if (!(moveptr->flags & LONG_CASTLE))
            return 0;

 /* test promotion */
    if (ptest->flags & PROMOTION) {
        if (!(moveptr->flags & PROMOTION))
            return 0;
        if (ptest->piece_taken != UNSPEC) {
            if (ptest->piece_taken != moveptr->prom)
                return 0;
        }
    }
 /* test mate */
    if (ptest->flags & MATE) {
        if (!test_check(1))
            return 0;
    }
 /* test check */
    if (ptest->flags & CHECK) {
        if (!test_check(0))
            return 0;
    }
    return 1;
}

/*
 * TEST_CAPTURE
 *
 * Test a capture
 */
static int
test_capture(ptest)
    register PTest ptest;
{
    register TestMove moveptr;
    int offset;

    moveptr = &testmoves[movenum];

    if (!(moveptr->flags & CAPTURE))
        return 0;

 /* test colour */
    if (ptest->colour != UNSPEC) {
        if ((u_char) to_colour(moveptr->halfmove) != ptest->colour)
            return 0;
    }
 /* test piece */
    if (ptest->piece != UNSPEC) {
        offset = to_offset(moveptr->file_from, moveptr->rank_from);
        if (ptest->piece != (u_char) (moveptr->board[offset] & 7))
            return 0;
    }
    if (ptest->piece_taken != UNSPEC && !(ptest->flags & PROMOTION)) {
        offset = to_offset(moveptr->file_to, moveptr->rank_to);
        if (ptest->piece_taken != (u_char) (moveptr->board[offset] & 7));
        return 0;
    }
 /* test square/file/rank captured from */
    if (ptest->file_from != UNSPEC) {
        if (ptest->file_from != moveptr->file_from)
            return 0;
    }
    if (ptest->rank_from != UNSPEC) {
        if (ptest->rank_from != moveptr->rank_from)
            return 0;
    }
 /* test square/file/rank captured to */
    if (ptest->file_to != UNSPEC) {
        if (ptest->file_to != moveptr->file_to)
            return 0;
    }
    if (ptest->rank_to != UNSPEC) {
        if (ptest->rank_to != moveptr->rank_to)
            return 0;
    }
 /* test en-passant */
    if (ptest->flags & ENPASSANT) {
        if (!(moveptr->flags & ENPASSANT))
            return 0;
    }
 /* test promotion */
    if (ptest->flags & PROMOTION) {
        if (!(moveptr->flags & PROMOTION))
            return 0;
        if (ptest->piece_taken != UNSPEC) {
            if (ptest->piece_taken != moveptr->prom)
                return 0;
        }
    }
 /* test mate */
    if (ptest->flags & MATE) {
        if (!test_check(1))
            return 0;
    }
 /* test check */
    if (ptest->flags & CHECK) {
        if (!test_check(0))
            return 0;
    }
    return 1;
}

/*
 * TEST_PASS_P
 *
 * Test for passed-pawns
 */
static int
test_pass_p(ptest)
    register PTest ptest;
{
    int i, file, rank, nfound;

    board = testmoves[movenum].board;
    nfound = 0;

 /* test white pawns */
    if (ptest->colour == 0 || ptest->colour == UNSPEC) {
        for (rank = 1; rank < 7; rank++) {
            for (file = 0; file < 8; file++) {
                if (board[to_offset(file, rank)] != PAWN)
                    continue;
                i = rank + 1;
                while (i < 7) {
                    if (file > 0) {
                        if (board[to_offset(file - 1, i)]
                          == (u_char) (PAWN | 8))
                            break;
                    }
                    if (board[to_offset(file, i)]
                      == (u_char) (PAWN | 8))
                        break;
                    if (file < 7) {
                        if (board[to_offset(file + 1, i)]
                          == (u_char) (PAWN | 8))
                            break;
                    }
                    i++;
                }
                if (i == 7)
                    nfound++;
            }
        }
    }
 /* test black pawns */
    if (ptest->colour == 1 || ptest->colour == UNSPEC) {
        for (rank = 6; rank > 0; rank--) {
            for (file = 0; file < 8; file++) {
                if (board[to_offset(file, rank)]
                  != (u_char) (PAWN | 8))
                    continue;
                i = rank - 1;
                while (i > 0) {
                    if (file > 0) {
                        if (board[to_offset(file - 1, i)]
                          == PAWN)
                            break;
                    }
                    if (board[to_offset(file, i)] == PAWN)
                        break;
                    if (file < 7) {
                        if (board[to_offset(file + 1, i)]
                          == PAWN)
                            break;
                    }
                    i--;
                }
                if (i == 0)
                    nfound++;
            }
        }
    }
    return test_value(ptest, nfound);
}

/*
 * TEST_UTD_P
 *
 * Test for united-pawns
 */
static int
test_utd_p(ptest)
    register PTest ptest;
{
    return 0;
}

/*
 * TEST_ISO_P
 *
 * Test for isolated-pawns
 */
static int
test_iso_p(ptest)
    register PTest ptest;
{
    return 0;
}

/*
 * TEST_DBL_P
 *
 * Test for doubled-pawns
 */
static int
test_dbl_p(ptest)
    register PTest ptest;
{
    return 0;
}

/*
 * TEST_DBLISO_P
 *
 * Test for doubled-isolated-pawns
 */
static int
test_dbliso_p(ptest)
    register PTest ptest;
{
    return 0;
}

/*
 * TEST_VALUE
 *
 * Test a value in a "struct ptest"
 */
static int
test_value(ptest, value)
    register PTest ptest;
    int value;
{

    switch (ptest->relop) {
    case RELOP_LT:
        return value < (int) ptest->val;
    case RELOP_GT:
        return value > (int) ptest->val;
    case RELOP_LE:
        return value <= (int) ptest->val;
    case RELOP_GE:
        return value >= (int) ptest->val;
    case RELOP_EQ:
        return value == (int) ptest->val;
    case RELOP_NE:
        return value != (int) ptest->val;
    case RELOP_NONE:
    default:
        return 0;
    }
}

/*
 * TEST_CHECK
 *
 * Test for check after the given move
 */
static int
test_check(mate)
    int mate;
{
    TestMove moveptr, nmoveptr;
    u_char saved_board[64];
    int saved_enpassant, tested_flag;

    if (mate)
        tested_flag = MATE_TESTED;
    else
        tested_flag = CHECK_TESTED;

    moveptr = &testmoves[movenum];
    if (!(moveptr->check_tested & tested_flag)) {
        moveptr->check_tested |= tested_flag;
        nmoveptr = &testmoves[movenum + 1];

    /* save internal values */
        copy_board(cb_board, saved_board);
        saved_enpassant = cb_enpassant;

    /* insert those after move */
        copy_board(nmoveptr->board, cb_board);
        cb_enpassant = nmoveptr->enpassant;

        if (mate) {
            if (is_mate(to_colour(moveptr->halfmove)))
                moveptr->flags |= MATE;
        } else {
            if (is_check(to_colour(moveptr->halfmove)))
                moveptr->flags |= CHECK;
        }

        copy_board(saved_board, cb_board);
        cb_enpassant = saved_enpassant;

    }
    if (mate)
        return moveptr->flags & MATE;
    return moveptr->flags & CHECK;
}


/*
 * REMOVE_EMPTIES
 *
 * Remove all empty keys in the keyfile.
 *
 * Returns 0 if no entries where removed, 1 if entries where removed, or -1
 * if an error occured.
 */
static int
remove_empties()
{
    KR block, index;
    u_long i;
    int j, found, ret;

    ret = 0;

    if (!quiet)
        output(".");

    for (i = 2L; i < blocknum; i++) {
        block = (KR) mps_getblk(i);
        if (!block)
            return -1;
        if (block->type == KR_KEY_TYPE && !block->data.k.index) {
            found = 0;
            block->type = KR_EMPTY_TYPE;
            block = (KR) mps_getblk(block->prev);
            if (!block)
                return -1;
            if (block->type == KR_KEY_TYPE) {
                if (block->data.k.index)
                    index = (KR) mps_getblk(block->data.k.index);
                else
                    index = NULL;
            } else {
                index = (KR) mps_getblk(1L);    /* master index */
            }

            while (index) {
                if (index->type != KR_INDEX_TYPE) {
                    error("not an index!!!");
                }
                for (j = 0; j < KR_ENTRY_MAX; j++) {
                    if (index->data.i.entry[j] ==
                      (i | SUBKEY_MASK)) {
                        index->data.i.entry[j] = 0L;
                        found++;
                        ret = 1;
                        break;
                    }
                }

                if (found) {
                    break;
                }
                if (!index->next)
                    return -1;
                index = (KR) mps_getblk(index->next);
            }
        }
    }
    if (tidy_index() < 0) {
        error("error tidying index blocks");
        ret = -1;
    }
    return ret;
}

/*
 * TIDY_INDEX
 *
 * Look through the blocks and move all index entries to the start of the
 * index blocks, and remove any empty index blocks.
 */
static int
tidy_index()
{
    u_long i, nextblock, prevblock;
    KR block, index1, index2;
    int ent1, ent2;

    for (i = 0; i < blocknum; i++) {
        block = (KR) mps_getblk(i);
        if (!block)
            return -1;
        if (block->type != KR_KEY_TYPE &&
          block->type != KR_MASTER_TYPE)
            continue;
        if (!block->next)
            continue;
        index1 = (KR) mps_getblk(block->next);
        if (!index1)
            return -1;
        index2 = index1;
        ent1 = 0;
        ent2 = 1;
        for (;;) {
            if (!index1->data.i.entry[ent1]) {
                index1->data.i.entry[ent1] =
                  index2->data.i.entry[ent2];
                index2->data.i.entry[ent2] = 0L;
            }
            if (index1->data.i.entry[ent1])
                ent1++;
            if (ent1 >= KR_ENTRY_MAX) {
                ent1 = 0;
                if (!index1->next)
                    break;
                index1 = (KR) mps_getblk(index1->next);
                if (!index1)
                    return -1;
            }
            if (++ent2 >= KR_ENTRY_MAX) {
                ent2 = 0;
                if (!index2->next)
                    break;
                index2 = (KR) mps_getblk(index2->next);
                if (!index2)
                    return -1;
            }
        }

    /* remove any empty index blocks now */
        block = (KR) mps_getblk(i);
        if (!block)
            return -1;
        nextblock = block->next;

        while (nextblock) {
            index1 = (KR) mps_getblk(nextblock);
            nextblock = index1->next;

            if (!index1)
                return -1;
            for (ent1 = 0; ent1 < KR_ENTRY_MAX; ent1++)
                if (index1->data.i.entry[ent1])
                    break;
            if (ent1 == KR_ENTRY_MAX) {
                prevblock = index1->prev;
                index1->type = KR_EMPTY_TYPE;
                index2 = (KR) mps_getblk(prevblock);
                if (!index2)
                    return -1;
                index2->next = nextblock;
                if (index2->type == KR_KEY_TYPE)
                    index2->data.k.index = nextblock;
                index2 = (KR) mps_getblk(nextblock);
                if (!index2)
                    return -1;
                index2->prev = prevblock;
            }
        }
    }
    return 0;
}

/*
 * COMPRESS
 *
 * Compress the keyfile.  Works by looking for all empty keyblocks and then
 * changing all references to that block (in all the other blocks).
 */
static int
compress()
{
    register u_long i, j;
    u_long curr, old, new;
    KR block;
    int k, is_key;

    curr = 2L;
    for (i = 2L; i < blocknum; i++) {
        block = (KR) mps_getblk(i);
        if (!block)
            return -1;
        if (block->type == KR_EMPTY_TYPE)
            continue;
        if (i != curr) {
            if (block->type == KR_KEY_TYPE) {
                is_key = 1;
                old = i | SUBKEY_MASK;
                new = curr | SUBKEY_MASK;
            } else {                   /* index block */
                is_key = 0;
            }
            for (j = 1L; j < blocknum; j++) {
                if (j == i)
                    continue;
                block = (KR) mps_getblk(j);
                if (!block)
                    return -1;
                if (block->type != KR_KEY_TYPE &&
                  block->type != KR_INDEX_TYPE)
                    continue;
                if (block->next == i)
                    block->next = curr;
                if (block->prev == i)
                    block->prev = curr;
                if (block->type == KR_KEY_TYPE) {
                    if (block->data.k.index == i)
                        block->data.k.index = curr;
                } else if (is_key) {   /* index block */
                    for (k = 0; k < KR_ENTRY_MAX; k++) {
                        if (block->data.i.entry[k] == old)
                            block->data.i.entry[k] = new;
                    }
                }
            }
        }
        curr++;
    }

    output("(saved %lu bytes)\n", (blocknum - curr) * 80L);
}


/*
 * ADD_TO_KINDEX
 *
 * Add an entry to a key's index block, allocating additional blocks as
 * required.
 */
static int
add_to_kindex(key, entry)
    Key key;
    u_long entry;
{
    u_long prevblock;
    KR index, prev;
    int i;

    if (key->iblock) {                 /* an index block already exists for
                                          key */
        index = (KR) mps_getblk(key->iblock);
        if (!index)
            return -1;
        for (i = 0; i < KR_ENTRY_MAX; i++) {
            if (!index->data.i.entry[i]) {
                index->data.i.entry[i] = entry;
                return 0;
            }
        }
    /* need a new index block */
        prevblock = key->iblock;
    } else {
        prevblock = key->block;
    }

 /* allocate an additional (or completely new) index block */
    prev = (KR) mps_getblk(prevblock);
    if (!prev)
        return -1;
    if (!key->iblock)                  /* first index block */
        prev->data.k.index = blocknum;
    key->iblock = blocknum;
    index = allocblk();
    if (!index)
        return -1;
    index->type = KR_INDEX_TYPE;
    index->prev = prevblock;
    prev->next = key->iblock;
    index->data.i.entry[0] = entry;
    return 0;
}

/*
 * ALLOCBLK
 *
 * Allocate a block from the MPS system using the block number in "blocknum"
 */
static KR
allocblk()
{
    KR kr_addr;

    kr_addr = (KR) mps_addblk(blocknum++);
    if (kr_addr)
        bzero((char *) kr_addr, sizeof(struct kr));
    return kr_addr;
}
