%{
/***************************************************************************
 * KPARSE.Y -- Parser for .ks files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version	Date		Comments
 * =======	====		========
 * 1.0		15/11/93	Initial.
 ***************************************************************************/

#include "global.h"
#include "kcompile.h"
#include "regexp.h"

#ifdef DEBUG
#define KPARSE_DEBUG
extern int debug;		/* in cbkey.c */
#endif

/* locals */
static u_char test_board[64]; 	/* board position */
static int brdrank = 7;   	/* index into test_board */
static char *cptr;
static regexp *kregexp;
static struct ptest ptest;
static int i;
static KStest tmptest;

/* externals */
extern char *source;		/* in cbkey.c */
extern char *yytext;		/* in kscan.l */
extern int yyleng;		/* in kscan.l */
extern int linenum;		/* in kscan.l */
extern u_char krelop;		/* in kscan.l */
extern u_char kcolour;		/* in kscan.l */
extern short kval;		/* in kscan.l */
extern short keco;		/* in kscan.l */

/* prototypes */
void init_parser P__((void));
%}

%union {
	Key key_type;
	KStest test_type;
	char *str_type;
	int int_type;
}

%type <key_type> key sibkeys
%type <test_type> test
%type <str_type> string
%type <int_type> boolean

%token KEY_WORD HMOVENUM
%token PLAYER_WORD SOURCE_WORD YEAR_WORD ECO_WORD
%token RESULT_WORD NMOVES_WORD EP_WORD COLOUR
%token PIECE FILE_LET RANK_NUM RELATIONAL_OP ECOCODE STRING
%token VALUE RESULTVAL SHORTCASTLE LONGCASTLE INFO_TESTS_ONLY
%token SUBKEY_RELATED ECHO_WORD BOOL_TRUE BOOL_FALSE
%token KSD_WORD QSD_WORD WSD_WORD BSD_WORD WSQ_WORD BSQ_WORD
%token PASS_P_WORD UTD_P_WORD ISO_P_WORD DBL_P_WORD DBLISO_P_WORD

%left '&' '|' ',' ';' HMOVENUM '!'

%start keyfile
%%

/*
 * keysource file
 */
keyfile
	: init sibkeys
		{ rootkey = $2; }
	;

/*
 * initialisation
 */
init	: /* empty */
	| init INFO_TESTS_ONLY boolean
		{ info_tests_only = $3; }
	| init SUBKEY_RELATED boolean
		{ subkey_related = $3; }
	| init ECHO_WORD
	;

/*
 * a list of sibling keys
 */
sibkeys
	: /* empty */
		{
		$$ = NULL;
		}
	| sibkeys key
		{
		$$ = add_sibling_key($1, $2);
		if (!$$)
			YYERROR;
		}
	;

/*
 * a key (containing an option test, and optional sub-keys)
 */
key
	: KEY_WORD string '{' test sibkeys '}'
		{
		if (!$2)
			YYERROR;
		$$ = add_key($2, $4, $5);
		if (!$$)
			YYERROR;
		}
	;

test
	: /* empty */
		{ $$ = NOTEST; }
	| board
		{ $$ = add_test(TEST_BOARD, (PTR)test_board); }
	| material
		{ $$ = add_test(TEST_MATERIAL, (PTR)&ptest); }
	| movement
		{ $$ = add_test(TEST_MOVEMENT, (PTR)&ptest); }
	| capture
		{ $$ = add_test(TEST_CAPTURE, (PTR)&ptest); }
	| player
		{ $$ = add_test(TEST_PLAYER, (PTR)kregexp); }
	| source
		{ $$ = add_test(TEST_SOURCE, (PTR)kregexp); }
	| year
		{ $$ = add_test(TEST_YEAR, (PTR)&ptest); }
	| eco
		{ $$ = add_test(TEST_ECO, (PTR)&ptest); }
	| result
		{ $$ = add_test(TEST_RESULT, (PTR)&ptest); }
	| nmoves
		{ $$ = add_test(TEST_NMOVES, (PTR)&ptest); }
	| pass_p
		{ $$ = add_test(TEST_PASS_P, (PTR)&ptest); }
	| utd_p
		{ $$ = add_test(TEST_UTD_P, (PTR)&ptest); }
	| iso_p
		{ $$ = add_test(TEST_ISO_P, (PTR)&ptest); }
	| dbl_p
		{ $$ = add_test(TEST_DBL_P, (PTR)&ptest); }
	| dbliso_p
		{ $$ = add_test(TEST_DBLISO_P, (PTR)&ptest); }
	| test ',' test
		{ 
		$$ = add_conn(CONN_THEN, $1, $3);
		if (!$$) YYERROR;
		}
	| test ';' test
		{
		$$ = add_conn(CONN_LATER, $1, $3);
		if (!$$) YYERROR;
		}
	| test '&' test
		{
		$$ = add_conn(CONN_AND, $1, $3);
		if (!$$) YYERROR;
		}
	| test '|' test
		{
		$$ = add_conn(CONN_OR, $1, $3);
		if (!$$) YYERROR;
		}
	| HMOVENUM test
		{
		tmptest = add_test(TEST_HMOVENUM, (PTR)kval);
		if (!tmptest) YYERROR;
		$$ = add_conn(CONN_AND, tmptest, $2);
		if (!$$) YYERROR;
		}
	| '!' test
		{
		($2)->negative = 1;
		$$ = $2
		}
	| '(' test ')'
		{ $$ = $2 }
	;

/*
 * a specific board set-up
 *
 * '[' string x 8 ']'
 */

board
	: '[' brdln brdln brdln brdln brdln brdln brdln brdln ']'
	;

brdln
	: STRING
		{
			u_char _piece;
			int _is_black;

			if (yyleng != 10) {
				yyerror("board-line string not 8 characters long");
				YYERROR;
			}

			for (cptr = yytext+1, i = 0; i < 8; cptr++, i++) {
				_is_black = 0;
				switch(*cptr) {
				case 'k':
					_is_black++;
				case 'K':
					_piece = KING;
					break;
				case 'q':
					_is_black++;
				case 'Q':
					_piece = QUEEN;
					break;
				case 'b':
					_is_black++;
				case 'B':
					_piece = BISHOP;
					break;
				case 'n':
					_is_black++;
				case 'N':
					_piece = KNIGHT;
					break;
				case 'r':
					_is_black++;
				case 'R':
					_piece = ROOK;
					break;
				case 'p':
					_is_black++;
				case 'P':
					_piece = PAWN;
					break;
				case '*':
					_piece = 0xff;
					break;
				case ' ':
				case '-':
					_piece = 0;
					break;
				default:
					yyerror("illegal piece in board line");
					YYERROR;
				}
				if (_is_black)
					_piece |= 8;
				test_board[to_offset(i, brdrank)] = _piece;
			}
			if (brdrank == 0)
				brdrank = 7;
			else
				brdrank--;
		}
	;

/*
 * material on the board
 *
 * colour piece square|file|rank [op value]
 * [colour|piece] square|file|rank
 * colour piece op value
 * colour|piece op value
 * colour piece boardside op value
 * colour|piece boardside op value
 */
material
	: colour piece sfr_from
	| colour sfr_from
	| piece sfr_from
	| colour piece sfr_from relational_op value 
	| colour sfr_from relational_op value
	| piece sfr_from relational_op value
	| colour piece relational_op value
	| piece relational_op value
	| colour relational_op value
	| colour piece boardside relational_op value
	| colour boardside relational_op value
	| piece boardside relational_op value
	;

/*
 * movement of a piece
 *
 * colour piece [square|file|rank] '-' [square|file|rank] ['+']
 * [colour|piece] [square|file|rank] '-' [square|file|rank] ['+']
 * square|file|rank '-' ['+']
 * '-' square|file|rank ['+']
 * [colour] castle ['+']
 * [colour|file] '=' [piece] ['+']
 */
movement
	: colour piece sfr_from '-' sfr_to check
	| colour piece sfr_from '-' check
	| colour piece '-' sfr_to check
	| colour piece '-' check
	| colour sfr_from '-' sfr_to check
	| colour sfr_from '-' check
	| colour '-' sfr_to check
	| colour '-' check
	| piece sfr_from '-' sfr_to check
	| piece sfr_from '-' check
	| piece '-' sfr_to check
	| piece '-' check
	| sfr_from '-' sfr_to check
	| sfr_from '-' check
	| '-' sfr_to check
	| colour castle check
	| castle check
	| colour file_from prom piece_taken check
	| colour file_from prom check
	| colour prom piece_taken check
	| colour prom prom check
	| file_from prom piece_taken check
	| file_from prom check
	| prom piece_taken check
	| prom check
	;

/*
 * a capture of a piece
 *
 * [colour piece] [square|file|rank] 'x' [piece] [square|file|rank] ['+']
 * [colour|piece] [square|file|rank] 'x' [piece] [square|file|rank] ['+']
 * 'x' piece [square|file|rank] ['+']
 * 'x' square|file|rank ['+']
 * [colour] [square|file|rank] 'x' [file] ep ['+']
 * [colour|file] 'x' [file] '=' [piece] ['+']
 */

capture
	: colour piece sfr_from 'x' piece_taken sfr_to check
	| colour piece sfr_from 'x' piece_taken check
	| colour piece sfr_from 'x' sfr_to check
	| colour piece sfr_from 'x' check
	| colour piece 'x' piece_taken sfr_to check
	| colour piece 'x' piece_taken check
	| colour piece 'x' sfr_to check
	| colour piece 'x' check
	| colour sfr_from 'x' piece_taken sfr_to check
	| colour sfr_from 'x' piece_taken check
	| colour sfr_from 'x' sfr_to check
	| colour sfr_from 'x' check
	| colour 'x' piece_taken sfr_to check
	| colour 'x' piece_taken check
	| colour 'x' sfr_to check
	| colour 'x' check
	| piece sfr_from 'x' piece_taken sfr_to check
	| piece sfr_from 'x' piece_taken check
	| piece sfr_from 'x' sfr_to check
	| piece sfr_from 'x' check
	| piece 'x' piece_taken sfr_to check
	| piece 'x' piece_taken check
	| piece 'x' sfr_to check
	| piece 'x' check
	| sfr_from 'x' piece_taken sfr_to check
	| sfr_from 'x' piece_taken check
	| sfr_from 'x' sfr_to check
	| sfr_from 'x' check
	| 'x' piece_taken sfr_to check
	| 'x' piece_taken check
	| 'x' sfr_to check
	| colour sfr_from 'x' file_to ep check
	| colour sfr_from 'x' ep check
	| colour 'x' file_to ep check
	| colour 'x' ep check
	| sfr_from 'x' file_to ep check
	| sfr_from 'x' ep check
	| 'x' file_to ep check
	| 'x' ep check
	| colour file_from 'x' file_to prom piece_taken check
	| colour file_from 'x' file_to prom check
	| colour file_from 'x' prom piece_taken check
	| colour file_from 'x' prom check
	| colour 'x' prom file_to piece_taken check
	| colour 'x' file_to prom check
	| colour 'x' prom piece_taken check
	| colour 'x' prom check
	| file_from 'x' file_to prom piece_taken check
	| file_from 'x' file_to prom check
	| file_from 'x' prom piece_taken check
	| file_from 'x' prom check
	| 'x' file_to prom piece_taken check
	| 'x' file_to prom check
	| 'x' prom piece_taken check
	| 'x' prom check
	;

/*
 * player name
 *
 * player "regexp"
 */
player
	: PLAYER_WORD regexp
	;

/*
 * source information
 *
 * source "regexp"
 */
source
	: SOURCE_WORD regexp
	;

/*
 * year
 *
 * year op value
 */
year
	: YEAR_WORD relational_op value
	;

/*
 * ECO code
 *
 * eco op eco-code
 */
eco
	: ECO_WORD relational_op ECOCODE
		{ ptest.val = keco; }
	;

/*
 * Result
 *
 * result 1-0|0-1|draw
 * result op value
 */
result
	: RESULT_WORD RESULTVAL
		{
		ptest.val = kval;
		ptest.relop = RELOP_EQ;
		}
	| RESULT_WORD relational_op value
	;

/*
 * Number of moves
 *
 * nmoves op value
 */
nmoves
	: NMOVES_WORD relational_op value
	;

/*
 * Passed pawns
 *
 * colour pass_p op value
 */
pass_p
	: colour PASS_P_WORD relational_op value
	;

/*
 * United pawns
 *
 * colour utd_p op value
 */
utd_p
	: colour UTD_P_WORD relational_op value
	;

/*
 * Isolated pawns
 *
 * colour iso_p op value
 */
iso_p
	: colour ISO_P_WORD relational_op value
	;

/*
 * Doubled pawns
 *
 * colour dbl_p op value
 */
dbl_p
	: colour DBL_P_WORD relational_op value
	;

/*
 * Doubled-isolated pawns
 *
 * colour dbliso_p op value
 */
dbliso_p
	: colour DBLISO_P_WORD relational_op value
	;

/*
 *


/*
 * square, file or rank moved to or captured
 */
sfr_to
	: file_to '?'
	| '?' rank_to
	| file_to rank_to
	;

file_to
	: FILE_LET
		{ ptest.file_to = *yytext - 'a'; }
	;

rank_to
	: RANK_NUM
		{ ptest.rank_to = *yytext - '1'; }
	;

/*
 * square, file or rank moved from or currently at
 */
sfr_from
	: file_from '?'
	| '?' rank_from
	| file_from rank_from
	;

file_from
	: FILE_LET
		{ ptest.file_from = *yytext - 'a'; }
	;

rank_from
	: RANK_NUM
		{ ptest.rank_from = *yytext - '1'; }
	;

/*
 * colour
 */
colour
	: COLOUR
		{ ptest.colour = kcolour; }
	;

/*
 * piece
 */
piece
	: PIECE
		{ ptest.piece = text2piece(*yytext); }
	;

/*
 * piece captured
 */
piece_taken
	: PIECE
		{ ptest.piece_taken = text2piece(*yytext); }
	;

/*
 * relational operator
 */
relational_op
	: RELATIONAL_OP
		{ ptest.relop = krelop; }
	;

/*
 * castle
 */
castle
	: SHORTCASTLE
		{ ptest.flags |= SHORT_CASTLE; }
	| LONGCASTLE
		{ ptest.flags |= LONG_CASTLE; }
	;

/*
 * check
 */
check
	: /* empty */
	| '+'
		{ ptest.flags |= CHECK; }
	| '#'
		{ ptest.flags |= MATE; }
	;

/*
 * enpassant
 */
ep
	: EP_WORD
		{ ptest.flags |= ENPASSANT; }
	;

/*
 * king/queen/white/black side
 */
boardside
	: KSD_WORD
		{ ptest.flags |= KINGSIDE; }
	| QSD_WORD
		{ ptest.flags |= QUEENSIDE; }
	| WSD_WORD
		{ ptest.flags |= WHITESIDE; }
	| BSD_WORD
		{ ptest.flags |= BLACKSIDE; }
	| WSQ_WORD
		{ ptest.flags |= WHITESQUARE; }
	| BSQ_WORD
		{ ptest.flags |= BLACKSQUARE; }
	;

/*
 * promotion
 */
prom
	: '='
		{ ptest.flags |= PROMOTION; }
	;

/*
 * string
 */
string
	: STRING
		{
			i = strlen(yytext) - 2;		/* excluding quotes */
			$$ = mem_alloc(i+1);
			if (!$$)
				YYERROR;
			strncpy($$, yytext+1, i);
		}
	;

/*
 * regular expression
 */
regexp
	: string
		{
		kregexp = regcomp($1);
		if (!kregexp) {
			yyerror("invalid regular expression");
			YYERROR;
		}
		free($1);
		}
	;
/*
 * value
 */
value
	: VALUE
		{ ptest.val = kval; }
	;

/*
 * boolean
 */
boolean
	: BOOL_TRUE
		{ $$ = 1; }
	| BOOL_FALSE
		{ $$ = 0; }
	;

%%

/*
 * print parsing error messages
 */
void
yyerror(s)
	char *s;
{
	if (*s == '*') {	/* used to mark internally generated errors */
		error("%s(%d): %s", source, linenum, ++s);
	} else if (*yytext == ')') {
		error("%s(%d): unmatched ')'", source, linenum);
	} else if (*yytext == '}') {
		error("%s(%d): spurious connector or too many '('",
		    source, linenum);
	} else if (*yytext == '\0') {
		error("%s: missing '}'", source);
	} else if (strcmp(yytext, "key") == 0) {
		error("%s(%d): sub-key connected to tests by ',', '&' or '|'",
		    source, linenum);
	} else {
		error("%s(%d): %s, at or near \"%s\"", source, linenum, s,
		    yytext);
	}
}

/*
 * initialise the parser variables
 */
void
init_parser()
{
	init_ptest(&ptest);
	linenum = 1;
}

