/***************************************************************************
 * KCOMPILE.C -- compiler functions for .ks files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          18/11/93        Initial.
 * 1.1          28/01/94        Added pawn tests.
 ***************************************************************************/

#include "global.h"
#include "kcompile.h"
#include <ctype.h>

#ifdef DEBUG
#define KCOMPILE_DEBUG
#endif

Key rootkey = NULL;                    /* very top of keyfile */
int info_tests_only = 0;               /* tests are on the game info only */
int subkey_related = 0;                /* subkey tests related to parent
                                          tests */
int remove_empty_keys = 0;             /* remove keys with no contents */
int compress_keyfile = 0;              /* compress keyfile */

static u_char key_name[KEY_NAME_LEN];

static u_char cvthex P__((char c));
static u_char *proc_key_name P__((char *name));
#ifdef DEBUG
static void print_key_name P__((u_char * name));
#endif

/*
 * ADD_KEY
 *
 * Add a main or sub key.
 */
Key
add_key(name, tests, childlist)
    char *name;
    KStest tests;
    Key childlist;
{
    register Key new;

    if (tests == NULL)
        return NULL;

    new = (Key) mem_alloc(sizeof(struct key));
    if (!new)
        return NULL;
    new->name = proc_key_name(name);
    if (tests == NOTEST)
        new->tests = NULL;
    else
        new->tests = tests;
    new->childlist = childlist;

    return new;
}

/*
 * ADD_SIBLING_KEY
 *
 * Add a sibling key
 */
Key
add_sibling_key(old, new)
    Key old, new;
{
    Key ret;

    if (old) {
        ret = old;
        while (old->siblist)
            old = old->siblist;
        old->siblist = new;
    } else {
        ret = new;
    }

    return ret;
}

/*
 * ADD_TEST
 *
 * Allocate a new kstest structure to hold test data.
 */
KStest
add_test(test, ptr)
    int test;
    PTR ptr;
{
    register KStest new;

    if (!(test > TEST_NONE && test <= TEST_MAX) || !ptr) {
        yyerror("*invalid or NULL test statement");
        return NULL;
    }
    new = (KStest) mem_alloc(sizeof(struct kstest));
    if (!new)
        return NULL;

    new->type = test;
    new->negative = 0;

 /* extract test data */

    switch (test) {
    case TEST_BOARD:
    /* 8 x 8 board */
        new->ptr = (PTR) mem_alloc(64);
        if (!new->ptr)
            goto failed;
        copy_board(ptr, new->ptr);
        break;
    case TEST_PLAYER:
    case TEST_SOURCE:
    /* already malloc'd memory */
    case TEST_HMOVENUM:
    /* a short value */
        new->ptr = ptr;
        break;
    default:
    /* ptest structure */
        new->ptr = (PTR) mem_alloc(sizeof(struct ptest));
        if (!new->ptr)
            goto failed;
        bcopy((char *) ptr, (char *) new->ptr, sizeof(struct ptest));
        init_ptest(ptr);
        break;
    }

    return new;

failed:
    if (new->ptr)
        free(new->ptr);
    free(new);
    return NULL;
}

/*
 * ADD_CONN
 *
 * Add a connector
 */
KStest
add_conn(conn, left, right)
    int conn;
    KStest left, right;
{
    register KStest new;

    if (!(conn > CONN_NONE && conn <= CONN_MAX) || !left || !right ||
      left == NOTEST || right == NOTEST) {
        yyerror("*invalid connector");
        return NULL;
    }
    new = (KStest) mem_alloc(sizeof(struct kstest));
    if (!new)
        return NULL;
    new->type = conn;
    new->left = left;
    new->right = right;

    return new;
}

/*
 * FREE_TEST
 *
 * Free memory used by a test or connector
 *
 * *** Recursive ***
 */
void
free_test(test)
    KStest test;
{
    if (!test)
        return;

    if (test->type & 0x80) {
    /* connector */
        free_test(test->left);
        free(test->left);
        free_test(test->right);
        free(test->right);
        free(test);
    } else {
    /* test */
        if (test->type != TEST_RESULT)
            free(test->ptr);
        free(test);
    }

}

/*
 * INIT_PTEST
 *
 * Reset the elements of a piece test structure
 */
void
init_ptest(ptest)
    PTest ptest;
{
    ptest->val = 0;
    ptest->colour = UNSPEC;
    ptest->piece = UNSPEC;
    ptest->piece_taken = UNSPEC;
    ptest->file_from = UNSPEC;
    ptest->rank_from = UNSPEC;
    ptest->file_to = UNSPEC;
    ptest->rank_to = UNSPEC;
    ptest->relop = RELOP_NONE;
    ptest->flags = 0;
}

/*
 * REGERROR
 *
 * Dummy function to stop regular expression error messages being written
 */
void
regerror()
{
 /* nothing */
}

/*
 * CVTHEX
 *
 * Convert a hex character to binary
 */
static u_char
cvthex(c)
    char c;
{
    if (c >= '0' && c <= '9')
        return c - '0';
    else if (c >= 'a' && c <= 'f')
        return (c - 'a') + 0x0a;
    else if (c >= 'A' && c <= 'F')
        return (c - 'A') + 0x0a;
    return 0;
}

/*
 * PROC_KEY_NAME
 *
 * Convert special characters in key name.
 */
static u_char *
proc_key_name(name)
    char *name;
{
    register char *cptr;
    int i;
    int backslash;
    u_char c;

    for (i = 0, cptr = name; i < KEY_NAME_LEN - 1, *cptr;) {
        c = 0;
        if (!backslash && *cptr == '\\' && *(cptr + 1) == 'x') {
            cptr += 2;
            if (*cptr && isxdigit(*cptr)) {
                c = cvthex(*cptr++);
                if (*cptr && isxdigit(*cptr)) {
                    c <<= 4;
                    c |= cvthex(*cptr++);
                }
            }
        } else if (*cptr == '%') {
            cptr++;
            switch (*cptr) {
            case 'k':
            case 'K':
                c = 177;
                break;
            case 'q':
            case 'Q':
                c = 178;
                break;
            case 'n':
            case 'N':
                c = 179;
                break;
            case 'b':
            case 'B':
                c = 180;
                break;
            case 'r':
            case 'R':
                c = 181;
                break;
            case 'p':
            case 'P':
                c = 182;
                break;
            default:
                c = 0;
                break;
            }
            if (c)
                cptr++;
        } else {
            c = *cptr++;
            backslash = c == '\\' ? 1 : 0;
        }

        if (c)
            key_name[i++] = c;
    }
    key_name[i++] = '\0';

 /* re-use space of input string */
    bcopy(key_name, name, i);
    return (u_char *) name;
}

#ifdef DEBUG

/*
 * PRINT_KEY_NAME
 *
 * Print the keyname, expanding binary digits to "\x??" form
 */
static void
print_key_name(name)
    u_char *name;
{
    if (!name) {
        fputs("<no name>", stdout);
        return;
    }
    fputc('"', stdout);
    while (*name) {
        if (*name >= 177 && *name <= 182)
            fputc(piece_list[*name - 176], stdout);
        else if (*name & 0x80 || *name < ' ')
            fprintf(stdout, "\\x%02x", (int) *name);
        else
            fputc(*name, stdout);
        name++;
    }
    fputc('"', stdout);
}

/*
 * DUMP_KEY
 *
 * dump the contents of a key and it's children and siblings
 */
void
dump_key(key, level)
    Key key;
    int level;
{
    if (!key) {
        fputs("null key!\n", stdout);
        return;
    }
    print_key_name(key->name);
    fprintf(stdout, " (level %d)\n", level);

    if (key->tests) {
        dump_test(key->tests);
        fputc('\n', stdout);
    }
    if (key->childlist)
        dump_key(key->childlist, level + 1);

    if (key->siblist)
        dump_key(key->siblist, level);
}

/*
 * DUMP_TESTS
 *
 * Dump the contents of a "struct kstest"
 */
void
dump_test(test)
    KStest test;
{
    char *msg;

    if (!test) {
        fputs("null test!\n", stdout);
        return;
    }
    if (is_test(test->type)) {
        if (test->negative)
            fputc('!', stdout);
    /* test */
        switch (test->type) {
        case TEST_BOARD:
            fputs("board ", stdout);
            break;
        case TEST_MATERIAL:
            fputs("material ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_MOVEMENT:
            fputs("movement ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_CAPTURE:
            fputs("capture ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_HMOVENUM:
            fputs("hmovenum ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_PLAYER:
            fputs("player ", stdout);
            break;
        case TEST_SOURCE:
            fputs("source ", stdout);
            break;
        case TEST_YEAR:
            fputs("year ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_ECO:
            fputs("eco ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_RESULT:
            fputs("result ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_NMOVES:
            fputs("nmoves ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_PASS_P:
            fputs("pass_p ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_UTD_P:
            fputs("utd_p ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_ISO_P:
            fputs("iso_p ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_DBL_P:
            fputs("dbl_p ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        case TEST_DBLISO_P:
            fputs("dbliso_p ", stdout);
            dump_ptest((PTest) test->ptr);
            break;
        default:
            fputs("{unknown test}", stdout);
            break;
        }
    } else {
    /* connector */
        fputc('(', stdout);
        dump_test(test->left);
        switch (test->type) {
        case CONN_AND:
            msg = " AND\n";
            break;
        case CONN_OR:
            msg = " OR\n";
            break;
        case CONN_THEN:
            msg = " THEN\n";
            break;
        case CONN_LATER:
            msg = " LATER\n";
            break;
        default:
            msg = " {unknown conn} ";
            break;
        }
        fputs(msg, stdout);
        dump_test(test->right);
        fputc(')', stdout);
    }
}

/*
 * DUMP_PTEST
 *
 * Dump the contents of a "struct ptest"
 */
void
dump_ptest(ptest)
    PTest ptest;
{
    char *msg;

    if (!ptest) {
        fputs("null ptest!\n", stdout);
        return;
    }
    fprintf(stdout, "value=%d ", (int) ptest->val);
    if (ptest->colour != UNSPEC)
        fputs(ptest->colour ? "Bl" : "Wh", stdout);
    if (ptest->piece != UNSPEC)
        fputc(dump_piece(ptest->piece), stdout);
    if (ptest->file_to != UNSPEC || ptest->rank_to != UNSPEC)
        fputs(" from=", stdout);
    if (ptest->file_from != UNSPEC)
        fputc(ptest->file_from + 'a', stdout);
    else
        fputc('?', stdout);
    if (ptest->rank_from != UNSPEC)
        fputc(ptest->rank_from + '1', stdout);
    else
        fputc('?', stdout);

    if (ptest->piece_taken != UNSPEC) {
        fputs("taking/promoting to:", stdout);
        fputc(dump_piece(ptest->piece_taken), stdout);
    }
    if (ptest->file_to != UNSPEC || ptest->rank_to != UNSPEC) {
        fputs(" to=", stdout);
        if (ptest->file_to != UNSPEC)
            fputc(ptest->file_to + 'a', stdout);
        else
            fputc('?', stdout);
        if (ptest->rank_to != UNSPEC)
            fputc(ptest->rank_to + '1', stdout);
        else
            fputc('?', stdout);
    }
    switch (ptest->relop) {
    case RELOP_NONE:
        msg = NULL;
        break;
    case RELOP_LT:
        msg = " <";
        break;
    case RELOP_GT:
        msg = " >";
        break;
    case RELOP_LE:
        msg = " <=";
        break;
    case RELOP_GE:
        msg = " >=";
        break;
    case RELOP_EQ:
        msg = " ==";
        break;
    case RELOP_NE:
        msg = " !=";
        break;
    default:
        msg = "{unknown relop}";
        break;
    }
    if (msg)
        fputs(msg, stdout);

    fputs(" flags=", stdout);
    if (ptest->flags & LONG_CASTLE)
        fputs(" o-o-o", stdout);
    if (ptest->flags & SHORT_CASTLE)
        fputs(" o-o", stdout);
    if (ptest->flags & ENPASSANT)
        fputs(" ep", stdout);
    if (ptest->flags & PROMOTION)
        fputs(" =", stdout);
    if (ptest->flags & CHECK)
        fputs(" +", stdout);
    if (ptest->flags & MATE)
        fputs(" #", stdout);
    if (ptest->flags & KINGSIDE)
        fputs(" ksd", stdout);
    if (ptest->flags & QUEENSIDE)
        fputs(" qsd", stdout);
    if (ptest->flags & WHITESIDE)
        fputs(" wsd", stdout);
    if (ptest->flags & BLACKSIDE)
        fputs(" bsd", stdout);
    if (ptest->flags & WHITESQUARE)
        fputs(" wsq", stdout);
    if (ptest->flags & BLACKSQUARE)
        fputs(" bsq", stdout);
}

#endif                                 /* DEBUG */
