/***************************************************************************
 * CKDUMP -- Dump the contents of a keyfile.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          11/11/93        Initial.
 * 1.1          08/02/94        Added -L option.  Removed indenting.
 ***************************************************************************/

#include "global.h"

#define BANNER "CKDUMP  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("Version 1.1 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("Version 1.1\n");
#endif

static int list = 0;                   /* -l and -L flags */
static int nogames = 0;                /* -L flag */
static struct game dump_game;

static void usage P__((void));
static void ck_dump P__((void));
static int ck_list P__((u_long recnum));

Database db;

static void
usage()
{
    error("usage: ckdump [options] ck-file");
    error("options:");
    error("  -l\t\tlist keyfile");
    error("  -L\t\tlist keyfile without game entries");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c;
    File keyfile;

    opterr = 0;

    while ((c = getopt(argc, argv, "lL")) != EOF) {
        switch (c) {
        case 'l':
            list++;
            break;
        case 'L':
            list++;
            nogames++;
            break;
        case '?':
        default:
            usage();
        }
    }

    argc -= optind;
    argv += optind;

    if (argc != 1)
        usage();

    output(BANNER);
    VERSION();

    if (mps_init(sizeof(struct kr)) < 0)
        return 1;

    keyfile = file_open(*argv, "rb");
    if (!keyfile)
        goto fail;

    if (kr_read_all(keyfile) < 0)
        goto fail;

    file_close(keyfile);

    if (list) {
		if (nogames)
			ck_list(0L);
		else {
        	db = open_database(*argv);
        	if (!db)
            	return 1;
	        ck_list(0L);
    	    close_database(db);
		}
    } else {
        ck_dump();
    }

    mps_cleanup();
    return 0;

fail:
    if (keyfile)
        file_close(keyfile);
    mps_cleanup();
    return 1;
}

static void
ck_dump()
{
    register u_long i;
    u_long nrecs;
    KR block;
    int j;

    block = (KR) mps_getblk(0L);
    if (!block)
        return;
    if (block->type != KR_MASTER_TYPE) {
        error("master record not at start of file");
        return;
    }
    nrecs = block->data.m.nrecs;

    output("%lu records\n", nrecs);

    for (i = 0; i < nrecs; i++) {
        block = (KR) mps_getblk(i);
        output("block #%lx type=%04x next=%08lx prev=%08lx\n",
          i, block->type, block->next, block->prev);
        switch (block->type) {
        case KR_MASTER_TYPE:
            output("(master record) nkeys=%08lx nrecs=%08lx lastgame=%08lx\n",
              block->data.m.nkeys, block->data.m.nrecs, block->data.m.lastgame);
            break;
        case KR_KEY_TYPE1:
		case KR_KEY_TYPE2:
            output("(%s record) index=%08lx name=%s\n",
			  block->type == KR_KEY_TYPE1 ? "old" : "new",
              block->data.k.index, block->data.k.name);
            break;
        case KR_INDEX_TYPE:
            output("(index record)\n");
            for (j = 0; j < KR_ENTRY_MAX; j++) {
                output("%08lx ", block->data.i.entry[j]);
                if (j == 5 || j == 11 || j == 17)
                    putchar('\n');
            }
            putchar('\n');
            break;
        case KR_EMPTY_TYPE:
            output("(empty record)\n");
            break;
        default:
            output("(ILLEGAL RECORD)\n");
            break;
        }
    }
}

static int
ck_list(recnum)
    u_long recnum;
{
    u_long i;
    u_long entry;
    KR_index index;
    KR kr_addr;

    kr_addr = (KR) mps_getblk(recnum);
    if (!kr_addr) {
        error("memory error");
        return -1;
    }
    if (kr_addr->type == KR_KEY_TYPE1 || kr_addr->type == KR_KEY_TYPE2) {
        output(cvt_sym((u_char *) kr_addr->data.k.name));
        newline();
    } else if (kr_addr->type != KR_MASTER_TYPE) {
        output("wrong type of record (%04x)\n", kr_addr->type);
        return -1;
    }
    while (kr_addr->next) {
        kr_addr = (KR) mps_getblk(kr_addr->next);       /* address of index
                                                           record */
        if (!kr_addr)
            return -1;
        index = &(kr_addr->data.i);
        for (i = 0; i < KR_ENTRY_MAX; i++) {
            entry = index->entry[i];
            if (entry) {
                if (entry & SUBKEY_MASK) {
                    if (ck_list(entry & ENTRY_MASK) < 0)
                        return -1;
                } else {
                    dump_game.num = entry;
                    if (!nogames) {
                    	if (read_info(db, &dump_game))
                        	continue;
                        output(format_info(&dump_game,
                            1));
                        newline();
                    }
                }
            }
        }
    }

    return 0;
}
