/***************************************************************************
 * CBTIDY -- Tidy databases.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          6/3/94          Initial.
 * 1.1			23/11/94		Additional options.
 ***************************************************************************/

#include "global.h"
#include <ctype.h>

#define BANNER "CBTIDY  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.1 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.1\n");
#endif

static int quiet = 0;                  /* -q flag */
static int informant = 0;              /* -i flag */
static int mslash = 0;                 /* -m flag */
static u_long first = 1L, last = 0xffffffffL;

static void usage P__((void));
static int tidy P__((Database ipdb, Database opdb));
static int remove_inf_num P__((Game game));
static int remove_mslash P__((Game game));

static void
usage()
{
    error("usage: cbtidy [options] input-database output-database");
    error("options:");
	error("  -g x\tset gap between games when writing");
    error("  -i\t\tremove informant number from source field");
    error("  -I\t\tignore game checksum errors :-)");
    error("  -m\t\tchange \"(m/X)\" round numbers to \"(X)\"");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to tidy");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;
    Database ipdb, opdb;
	char *cptr;

    opterr = 0;
    while ((c = getopt(argc, argv, "g:iImqr:")) != EOF) {
        switch (c) {
		case 'g':
			write_gap = (int)strtol(optarg, &cptr, 10);
			if (*cptr) {
				error("invalid gap value \"%s\"", optarg);
				return 1;
			}
			break;
        case 'i':
            informant++;
            break;
        case 'I':
            ignore_chksum_err++;
            break;
        case 'm':
            mslash++;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (argc != 2)
        usage();

    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    kill_ext(*argv);
    ipdb = open_database(*argv);
    if (!ipdb)
        return 1;

    argv++;
    kill_ext(*argv);
    no_error = 1;
    opdb = open_database(*argv);
    no_error = 0;
    if (!opdb) {
        if (!quiet)
            output("creating database %s\n", *argv);
        opdb = create_database(*argv);
        if (!opdb) {
            close_database(ipdb);
            return 1;
        }
    }
    ret = tidy(ipdb, opdb);

    close_database(opdb);
    close_database(ipdb);
    return ret;
}

/*
 * TIDY
 *
 * Perform tidying operation on database
 */

static int
tidy(ipdb, opdb)
    Database ipdb, opdb;
{
    Game game;
    u_long i, ngames;
    int ret, progress, last_report;

    if (last > ipdb->ngames)
        last = ipdb->ngames;
    if (first > last)
        first = last;

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game)
        return 1;

    last_report = -1;

    ngames = (last - first) + 1L;
    ret = 0;

    file_seek(opdb->cbf, read_index(opdb, opdb->ngames + 1L));

    for (i = first; i <= last; i++) {
        game->num = i;
        if (read_game(ipdb, game) < 0)
            continue;
        if (is_deleted(game->header))
            continue;

#if 0
        printf("before \"%s\" len=%d\n", game->sinfo,
          (int) game->slen);
#endif

        if (informant)
            remove_inf_num(game);

        if (mslash)
            remove_mslash(game);

#if 0
        printf("after \"%s\" len=%d\n", game->sinfo,
          (int) game->slen);
#endif

        game->num = opdb->ngames + 1L;
        if (write_game(opdb, game) < 0) {
            ret = 1;
            goto quit;
        }
        opdb->ngames++;

    /* report progess of operation */
        if (!quiet) {
            progress = (int) ((i * 100L) / ngames);
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
        game_tidy(game);
    }
quit:
    write_ngames(opdb);
    game_free(game);
    return ret;
}

/*
 * REMOVE_INF_NUM
 *
 * Remove the informant number "x/yyy" or "x/(yyy)" from the source field.
 */
static int
remove_inf_num(game)
    Game game;
{
    char *cptr, *anchor;
    int quotes;

    if (!game->slen)
        return 0;
    cptr = (char *) game->sinfo;
    while (*cptr) {
        if (isdigit(*cptr)) {
            anchor = cptr;
            while (isdigit(*cptr))
                cptr++;
            if (*cptr++ != '/')
                continue;
            if (*cptr == '(') {
                quotes = 1;
                cptr++;
            } else {
                quotes = 0;
            }
            if (!isdigit(*cptr))
                continue;
            while (isdigit(*cptr))
                cptr++;
            if (quotes)
                if (*cptr++ != ')')
                    continue;
            if (*cptr && *cptr != ' ')
                continue;

        /* now remove the text */
            while (*anchor++ = *cptr++);

            tidy_string((char *) game->sinfo);
            game->slen = ustrlen(game->sinfo);
            return 1;
        }
        cptr++;
    }
    return 0;
}

/*
 * REMOVE_MSLASH
 *
 * Change (m/XX) round numbering to (XX).
 */
static int
remove_mslash(game)
    Game game;
{
    char *cptr, *anchor;

    if (!game->slen)
        return 0;
    cptr = (char *) game->sinfo;
    while (*cptr) {
        if (*cptr == '(') {
            anchor = cptr;
            cptr++;
            if (*cptr != 'm' || *(cptr + 1) != '/')
                continue;
            cptr += 2;
            if (!isdigit(*cptr))
                continue;
            while (isdigit(*cptr))
                cptr++;
            if (*cptr != ')')
                continue;
            cptr++;

        /* now replace the "(m/" with just "(" */
            *anchor++ = ' ';
            *anchor++ = ' ';
            *anchor = '(';

            tidy_string((char *) game->sinfo);
            game->slen = ustrlen(game->sinfo);
            return 1;
        }
        cptr++;
    }
    return 0;
}
