/***************************************************************************
 * CBSPLIT -- Split databases by year.
 *
 * Copyright (c)1994 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          06/03/94        Initial.
 ***************************************************************************/

#include "global.h"

#define BANNER "CBSPLIT  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.0 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.0\n");
#endif

static int quiet = 0;                  /* -q flag */
static u_long first = 1L, last = 0xffffffffL;
static char *listfile = NULL;
static char *yeardir = NULL;

static void usage P__((void));
static int split P__((char *ipname));

static void
usage()
{
    error("usage: cbsplit [options] -d yearbase-directory dbspec [... dbspec]");
    error("options:");
    error("  -f listfile\tread from list of additional databases to split");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to merge (slow merge only)");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;
    char *name;
    FILE *lfp;

    opterr = 0;
    while ((c = getopt(argc, argv, ":d:f:qr:")) != EOF) {
        switch (c) {
        case 'f':
            listfile = optarg;
            break;
        case 'd':
            yeardir = optarg;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (!listfile && !argc)
        usage();

    if (!yeardir) {
        error("you must specify a \"yeardir\" with -d");
        return 1;
    }
    if (isdir(yeardir) != 1) {
        error("\"%s\" is not a directory", yeardir);
        return 1;
    }
    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    ret = 0;
    while (argc && !ret) {
        if (find_database_init(*argv) < 0)
            return 1;
        while (name = find_database()) {
            if (split(name) < 0) {
                ret = 1;
                break;
            }
        }
        find_database_exit();
        argc--;
        argv++;
    }

    if (listfile && !ret) {
        lfp = fopen(listfile, "rt");
        if (!lfp) {
            error("error opening listfile \"%s\"", listfile);
        } else {
            while (name = read_listfile(lfp)) {
                if (split(name) < 0) {
                    ret = 1;
                    break;
                }
            }
            fclose(lfp);
        }
    }
    return ret;
}

/*
 * SPLIT
 *
 * Open an input database, and write each game into an output database
 * based on the year of the game.
 */
static int
split(ipname)
    char *ipname;
{
    static char opname[1024];
    u_long i, dbfirst, dblast;
    int curryear = -1;
    Database ipdb, opdb;
    Game game;
    int ret;

    kill_ext(ipname);
    ipdb = open_database(ipname);
    if (!ipdb)
        return -1;

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game) {
        close_database(ipdb);
        return -1;
    }
    if (!quiet)
        output("splitting %s (%lu games)\n", ipname, ipdb->ngames);

    if (last > ipdb->ngames)
        dblast = ipdb->ngames;
    else
        dblast = last;
    if (first > dblast)
        dbfirst = dblast;
    else
        dbfirst = first;

    ret = 0;

    for (i = dbfirst; i <= dblast; i++) {
        game->num = i;
        if (read_game(ipdb, game) < 0)
            continue;
        if (is_deleted(game->header))
            continue;

        if (curryear != (int) game->year) {
            if (curryear != -1)
                close_database(opdb);
            curryear = (int) game->year;
            sprintf(opname, "%s%c%04dbase", yeardir, PATHSEP,
              game->year);
            no_error = 1;
            opdb = open_database(opname);
            no_error = 0;
            if (!opdb) {
                opdb = create_database(opname);
                if (!opdb) {
                    ret = -1;
                    goto quit;
                }
            }
            file_seek(opdb->cbf,
              read_index(opdb, opdb->ngames + 1L));
        }
        game->num = opdb->ngames + 1L;
        if (write_game(opdb, game) < 0) {
            ret = -1;
            goto quit;
        }
        opdb->ngames++;
        write_ngames(opdb);
        game_tidy(game);
    }
quit:
    close_database(opdb);
    close_database(ipdb);
    game_free(game);
    return ret;
}
