/***************************************************************************
 * CBSORT -- Sort games in a ChessBase file.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 0.8          11/11/93        Initial.
 * 1.0          05/03/94        Added "round number" sorting and file list
 *                              processing.
 ***************************************************************************/

#include "global.h"

#define BANNER "CBSORT  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.0 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.0\n");
#endif

static int quiet = 0;                  /* -q flag */
static char *dbname;
static char *listfile = NULL;
static u_long first = 1L, last = 0xffffffffL;
static u_long saved_first, saved_last;
static Database db = NULL;
static Game game = NULL;
static Order order[MAX_ORDER] = {YEAR, SOURCE, ROUND, W_PLAYER, B_PLAYER,
	NOORDER};
static Order left[MAX_ORDER] = { W_PLAYER, B_PLAYER, SOURCE,
	ECO, YEAR, NMOVES, ROUND, NOORDER};
struct game_sort ref;                  /* reference game for quicksort() and
                                          sort_games() */

/*
 * internal stack variables and macros used by quicksort()
 */
#define STACK_SZ 32
#define STACK_TYPE long
static STACK_TYPE stack[STACK_SZ];
static int stack_items;
#define stackinit() stack_items = 0
#define stackempty() !stack_items
#define push(x) { if (stack_items >= STACK_SZ) \
                        error("internal stack overflow"); \
                   else \
                        stack[stack_items++] = (x); }
#define pop() stack[--stack_items]

static void usage P__((void));
static int process_sortkey_arg P__((char *sortkey));
static int sort_reorg P__((void));
static void quicksort P__((long l, long r));
static int sort_games P__((void));
static int reorg_database P__((void));

static void
usage()
{
    error("usage: cbsort [options] database [... database]");
    error("options:");
    error("  -f listfile\tread from list of additional databases to sort");
    error("  -k\t\tspecify sort-key (default = \"ysrwb\")");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to sort");
    error("sort-key is one or more of:");
    error("  w\t\tWhite player name");
    error("  b\t\tBlack player name");
    error("  s\t\tSource information");
    error("  r\t\tRound number");
    error("  e\t\tECO code");
    error("  y\t\tYear");
    error("  n\t\tNumber of moves");
    exit(1);
}

/*
 * PROCESS_SORTKEY_ARG
 *
 * Process the -k (sort key) argument.
 */
static int
process_sortkey_arg(sortkey)
    char *sortkey;
{
    int i, c;
    Order key;

    i = 0;
    while (*sortkey && i < MAX_ORDER - 1) {
        switch (*sortkey) {
        case 'w':
        case 'W':
            key = W_PLAYER;
            break;
        case 'b':
        case 'B':
            key = B_PLAYER;
            break;
        case 's':
        case 'S':
            key = SOURCE;
            break;
        case 'e':
        case 'E':
            key = ECO;
            break;
        case 'y':
        case 'Y':
            key = YEAR;
            break;
        case 'n':
        case 'N':
            key = NMOVES;
            break;
        case 'r':
        case 'R':
            key = ROUND;
            break;
        default:
            error("unknown sortkey '%c'", *sortkey);
            return 1;
        }
        for (c = 0; c < MAX_ORDER; c++)
            if (left[c] == key)
                break;
        if (c == MAX_ORDER) {
            error("duplicate key '%c' in sortkey", *sortkey);
            return 1;
        }
        left[c] = NOORDER;
        order[i++] = key;

        sortkey++;
    }
    if (*sortkey) {
        error("too many keys in sortkey");
        return 1;
    }
    order[i] = NOORDER;
    return 0;
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;
    FILE *lfp;

    opterr = 0;
    while ((c = getopt(argc, argv, "f:k:qr:")) != EOF) {
        switch (c) {
        case 'f':
            listfile = optarg;
            break;
        case 'k':
            if (process_sortkey_arg(optarg))
                return 1;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (!listfile && !argc)
        usage();

    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    saved_first = first;
    saved_last = last;

    ret = 0;
    while (argc && !ret) {
        dbname = *argv;
        ret = sort_reorg();
        argc--;
        argv++;
    }

    if (listfile && !ret) {
        lfp = fopen(listfile, "rt");
        if (!lfp) {
            error("error opening listfile \"%s\"", listfile);
        } else {
            while (dbname = read_listfile(lfp)) {
                ret = sort_reorg();
                if (ret)
                    break;
            }
            fclose(lfp);
        }
    }
    return ret;
}

/*
 * SORT_REORG
 *
 * Perform sort and database reorganization
 */
static int
sort_reorg()
{
    int ret;

    ret = 0;
    db = open_database(dbname);
    if (!db)
        return 1;

    first = saved_first;
    last = saved_last;
    if (last > db->ngames)
        last = db->ngames;
    if (first > last)
        first = last;

    if (sort_games() < 0)
        ret = 1;
    else if (reorg_database() < 0)
        ret = 1;
    close_database(db);
    return ret;
}

static int progress, last_report;

/*
 * QUICKSORT
 *
 * Sort a range of game_sort structures using a quicksort algorithm.
 */
static void
quicksort(l, r)
    long l, r;
{
    register long i, j;

    if (r - l < 2)
        return;

    stackinit();
    for (;;) {
        while (r > l) {
            if (l < 0 || r < 0)
                break;
            i = l - 1;
            j = r;
            mps_fromblk(r, &ref);

            for (;;) {
                while (cmp_game((GameSort) mps_getblk(++i), &ref, order) < 0)
                    if (i >= j)
                        break;
                while (cmp_game((GameSort) mps_getblk(--j), &ref, order) > 0)
                    if (j <= 0)
                        break;
                if (i >= j)
                    break;
                mps_swapblk(i, j);
            }
            mps_swapblk(i, r);

            if (i - l > r - i) {
                push(l);
                push(i - 1);
                l = i + 1;
            } else {
                push(i + 1);
                push(r);
                r = i - 1;
            }
        }

        if (stackempty())
            break;
        r = pop();
        l = pop();
    }
}

/*
 * SORT_GAMES
 *
 * Sort games within a database.
 */
static int
sort_games()
{
    register u_long i;
    u_long im_base, im_num, ngames;
    int ret;
    GameSort game;

    ret = 0;
    last_report = -1;
    ngames = (last - first) + 1L;

    if (mps_init(sizeof(struct game_sort)) < 0)
        return -1;

 /* allocate as much memory as possible */
    for (im_num = 0; im_num < ngames; im_num++) {
        if (mps_addblk(im_num) == NULL)
            break;
    }

    if (im_num < ngames) {
        error("too many games to sort (%ld); only enough memory for %ld",
          ngames, im_num);
        mps_cleanup();
        return -1;
    }
    im_base = first;

    if (!quiet)
        output("reading %s\n", dbname);
    for (i = 0; i < im_num; i++) {
        game = (GameSort) mps_getblk(i);
        if (!game) {
            ret = -1;
            goto quit;
        }
        if (read_game_sort(db, im_base + i, game) < 0) {
            ret = -1;
            goto quit;
        }
        if (!quiet) {
            progress = (int) ((i * 100L) / im_num);
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
    }

 /* sort games in memory */
    if (!quiet)
        output("sorting... ");
    quicksort(0, im_num - 1);
    if (!quiet)
        output("done\n");

 /* write-out indexes (not game data) */
    if (!quiet)
        output("writing index entries\n");
    for (i = 0; i < im_num; i++) {
        game = (GameSort) mps_getblk(i);
        if (!game) {
            ret = -1;
            goto quit;
        }
        write_index(db, im_base + i, game->index);

        if (!quiet) {
            progress = (int) ((i * 100L) / im_num);
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
    }

quit:
    if (mps_cleanup() < 0)
        ret = -1;
    return ret;
}

/*
 * REORG_DATABASE
 *
 * Reorganise a database
 */
static int
reorg_database()
{
    register u_long i;
    u_long ngames;
    char *oldname;
    Database new_db;
    int ret;

    ret = 0;
    last_report = -1;

    oldname = xstrdup(db->cbf->name);
    if (!oldname)
        return -1;
 /* make a copy of the existing database */
    if (rename_database(db, "backup") < 0) {
        no_error = 1;
        close_database(db);
        db = NULL;
        no_error = 0;
        free(oldname);
        return -1;
    }
    new_db = db;
 /* recreate original database */
    db = create_database(oldname);
    if (!db) {
        ret = -1;
        goto quit;
    }
    output("reorganizing the database\n");
    game = (Game) mem_alloc(sizeof(struct game));
    if (!game) {
        ret = -1;
        goto quit;
    }
 /* read the games from the database copy and write them, in order, back into
    the existing database */

    file_seek(db->cbf, read_index(db, 1L));
    ngames = new_db->ngames;
    for (i = 1L; i <= ngames; i++) {

        if (!quiet) {
            progress = (int) ((i * 100L) / ngames);
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
        game->num = i;
        if (read_game(new_db, game) < 0) {
            ret = -1;
            goto quit;
        }
        if (write_game(db, game) < 0) {
            ret = -1;
            goto quit;
        }
        db->ngames++;
        game_tidy(game);
    }

    write_ngames(db);

quit:
    free(oldname);
    game_free(game);
    if (!ret) {
        no_error = 1;
        delete_database(new_db);
        no_error = 0;
    } else {
        close_database(new_db);
        output("errors... original database is in backup.cbf; use\n");
        output("CBFRESH to create a new index file *BEFORE* using\n");
        output("ChessBase on the file!\n");
    }
    return ret;
}
