/***************************************************************************
 * CBMERGE -- Merge databases.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          6/1/94          Initial.
 * 1.1			26/2/95			Added -I option
 ***************************************************************************/

#include "global.h"

#define BANNER "CBMERGE  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.1 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.1\n");
#endif

#define COPYBUF_SZ      24576          /* size of fast_merge() copy buffer */

static int quiet = 0;                  /* -q flag */
static int ignore_err = 0;			   /* -I flag */
static int fast = 1;                   /* do fast merge */
static u_long first = 1L, last = 0xffffffffL;
static char *opdbname = NULL;
static char *listfile = NULL;
static Database opdb = NULL;
static u_char *copybuf;
static int (*merge) P__((char *dbname));

static void usage P__((void));
static int slow_merge P__((char *dbname));
static int fast_merge P__((char *dbname));

static void
usage()
{
    error("usage: cbmerge [options] -o output-database dbspec [... dbspec]");
    error("options:");
    error("  -f listfile\tread from list of additional databases to merge");
	error("  -I Ignore input-database open errors");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to merge (slow merge only)");
    error("  -s\t\tslow merge (with error checking)");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;
    char *name;
    FILE *lfp;

    opterr = 0;
    while ((c = getopt(argc, argv, "qr:o:f:Ist")) != EOF) {
        switch (c) {
        case 'f':
            listfile = optarg;
            break;
		case 'I':
			ignore_err++;
			break;
        case 'o':
            opdbname = optarg;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case 's':
            fast = 0;
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (!listfile && !argc)
        usage();

    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    if (!opdbname) {
        error("no output database specified");
        return 1;
    }
    kill_ext(opdbname);
    no_error = 1;
    opdb = open_database(opdbname);
    no_error = 0;
    if (!opdb) {
        if (!quiet)
            output("creating database %s\n", opdbname);
        opdb = create_database(opdbname);
        if (!opdb)
            return 1;
    }
 /* allocate memory for fast_merge() copy buffer */
    if (fast) {
        copybuf = (u_char *) mem_alloc(COPYBUF_SZ);
        if (!copybuf) {
            close_database(opdb);
            return 1;
        }
    }
    if (fast)
        merge = fast_merge;
    else
        merge = slow_merge;

    ret = 0;
    while (argc) {
        if (find_database_init(*argv) < 0)
            return 1;
        while (name = find_database()) {
            if (merge(name) < 0) {
                ret = 1;
                break;
            }
        }
        find_database_exit();
        argc--;
        argv++;
    }

    if (listfile) {
        lfp = fopen(listfile, "rt");
        if (!lfp) {
            error("error opening listfile \"%s\"", listfile);
        } else {
            while (name = read_listfile(lfp)) {
                if (merge(name) < 0) {
                    ret = 1;
                    break;
                }
            }
            fclose(lfp);
        }
    }
    if (fast)
        free(copybuf);
    close_database(opdb);
    return ret;
}

/*
 * SLOW_MERGE
 *
 * Merge the named database into the output database "opdb".
 * This version reads each game and in so doing so validates the input
 * data.  It will also ignore games that have been marked as deleted.
 */
static int
slow_merge(dbname)
    char *dbname;
{
    Database db;
    Game game;
    u_long i, ngames, dbfirst, dblast;
    int ret, progress, last_report;

    if (strcmp(dbname, opdbname) == 0) {
        error("ignoring %s; this is the output database!", dbname);
        return 0;
    }
    db = open_database(dbname);
    if (!db)
        return ignore_err ? 0 : -1;

    if (last > db->ngames)
        dblast = db->ngames;
    else
        dblast = last;
    if (first > dblast)
        dbfirst = dblast;
    else
        dbfirst = first;

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game) {
        close_database(db);
        return -1;
    }
    last_report = -1;
    ngames = (dblast - dbfirst) + 1L;
    ret = 0;

    file_seek(opdb->cbf, read_index(opdb, opdb->ngames + 1L));

    if (!quiet)
        output("merging %s (%lu games)\n", dbname, db->ngames);

    for (i = dbfirst; i <= dblast; i++) {
        game->num = i;
        if (read_game(db, game) < 0)
            continue;
        if (is_deleted(game->header))
            continue;
        game->num = opdb->ngames + 1L;
        if (write_game(opdb, game) < 0) {
            ret = -1;
            goto quit;
        }
        opdb->ngames++;

    /* report progess of operation */
        if (!quiet) {
            progress = (int) ((i * 100L) / (ngames));
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
        game_tidy(game);
    }
quit:
    write_ngames(opdb);
    game_free(game);
    close_database(db);
    return ret;
}

/*
 * FAST_MERGE
 *
 * Merge the named database into the output database "opdb".
 * This version reads chunks of the input .cbf file onto the end of
 * of the output .cbf, and then writes the adjusted index entries to
 * the output .cbi file.  No game error checking can be done, and games
 * marked as deleted cannot be ignored.
 */
static int
fast_merge(dbname)
    char *dbname;
{
    Database db;
    u_long i, offset, new_offset, op_ngames, ngames;
    u_long filesize, total;
    int nread;

    if (strcmp(dbname, opdbname) == 0) {
        error("ignoring %s; this is the output database!", dbname);
        return 0;
    }
    db = open_database(dbname);
    if (!db)
        return ignore_err ? 0 : -1;

    op_ngames = opdb->ngames;
    offset = read_index(opdb, op_ngames + 1L);
    file_seek(opdb->cbf, offset);
    file_seek(db->cbf, read_index(db, 1L));
    total = 0L;
    filesize = file_length(db->cbf->fd);

    if (!quiet)
        output("merging %s (%lu games)\n", dbname, db->ngames);

    do {
        nread = file_read(db->cbf, (char *) copybuf, COPYBUF_SZ);
        if (nread == -1)
            break;
        file_write(opdb->cbf, (char *) copybuf, nread);
        total += (u_long) nread;
        if (!quiet)
            output("%d%%\r", (int) ((total * 100L) / filesize));
    } while (nread == COPYBUF_SZ);

    ngames = db->ngames;
    for (i = 1L; i <= ngames + 1L; i++) {
        new_offset = read_index(db, i);
        new_offset += offset;
        write_index(opdb, op_ngames + i, new_offset);
    }
    opdb->ngames += ngames;
    write_ngames(opdb);
    close_database(db);
    return 0;
}
