/***************************************************************************
 * CBKEY -- Generate keyfiles for ChessBase files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0a         04/07/94        Fixed "result" parsing error in kparse.y.
 * 1.0          11/11/93        Initial.
 ***************************************************************************/

#include "global.h"
#include "kcompile.h"

#define BANNER "CBKEY  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.0a [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.0a\n");
#endif

void usage P__((void));

#ifdef DEBUG
int debug = 0;                         /* -d flag */
#endif
int nogen = 0;                         /* -n flag */
int quiet = 0;                         /* -q flag */
char *ext, *source, *kfname;
File keyfile;
u_long first = 1L, last = 0xffffffffL;
Database db = NULL;

extern FILE *yyin;                     /* in kscan.l */
extern int yyparse P__((void));        /* in kparse.y */
extern void init_parser P__((void));   /* in kparse.y */

extern int classify_games P__((void)); /* in ktests.c */

#ifdef DEBUG
#define GETOPT_STR "cdeqr:t"
#else
#define GETOPT_STR "ceqr:t"
#endif

void
usage()
{
    error("usage: cbkey [options] source [keyfile]");
    error("options:");
    error("  -c\t\tcompress keyfile before writing (turns on -e)");
#ifdef DEBUG
    error("  -d\t\tenable debugging (to stdout)");
#endif
    error("  -e\t\tremove empty keys before writing");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to classify");
    error("  -t\t\tdon't generate keyfile; test source compilation only");
    error("source is a keysource file containing key specifications.");
    error("keyfile is in the form \"database.ext\" where \"database\"");
    error("is the path to the database, and \"ext\" is one of:");
    error("   ckp (players keyfile)");
    error("   ckm (middlegame keyfile)");
    error("   cke (endgame keyfile)");
    error("   ckt (source keyfile)");
    error("   cku (user keyfile)");
    error("note: openings keyfiles (.cko) cannot be created");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;

    ret = 0;

    opterr = 0;
    while ((c = getopt(argc, argv, GETOPT_STR)) != EOF) {
        switch (c) {
        case 'c':
            compress_keyfile = 1;
            remove_empty_keys = 1;
            break;
#ifdef DEBUG
        case 'd':
            debug++;
            break;
#endif
        case 'e':
            remove_empty_keys = 1;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case 't':
            nogen++;
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if ((nogen && argc != 1) || (!nogen && argc != 2))
        usage();

    source = *argv++;
    if (!nogen) {
        kfname = *argv;
        ext = kill_ext(kfname);
    }
    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    if (!nogen) {
    /* check the keyfile extension */
        if (!ext) {
            error("keyfile extension not specified");
            return 1;
        }
        if (strlen(ext) != 3 || *ext != 'c' || *(ext + 1) != 'k') {
            error("invalid keyfile extension specified");
            return 1;
        }
        c = *(ext + 2);
        if (c != 'p' && c != 'm' && c != 'e' && c != 't' && c != 'u') {
            error("unknown or unsupported keyfile type");
            return 1;
        }
    /* rebuild the source name and open the output file */
        *(ext - 1) = '.';
        keyfile = file_open(kfname, "wb");
        if (!keyfile)
            return 1;
    }
    if (strcmp(source, "-") == 0)
        yyin = stdin;
    else
        yyin = fopen(source, "rt");
    if (!yyin) {
        error("couldn't open source file \"%s\"", source);
        return 1;
    }
    if (!nogen) {
        db = open_database(kfname);
        if (!db)
            return 1;


        if (last > db->ngames)
            last = db->ngames;
        if (first > last)
            first = last;
    }
 /* compile the keysource file */
    if (!quiet)
        output("compiling %s\n", source);
    rootkey = NULL;
    init_parser();
    ret = yyparse();
    if (ret)
        error("error compiling %s\n", source);
    else if (!rootkey)
        error("no key definitions in %s\n", source);

#ifdef DEBUG
    if (!ret && debug && rootkey)
        dump_key(rootkey, 0);
#endif

    if (!nogen) {
        if (!ret) {
            if (!quiet)
                output("classifying games\n");
            if (classify_games() < 0)
                ret = 1;
        }
        file_close(keyfile);
        close_database(db);
    }
    return ret;
}
