/***************************************************************************
 * CBGEN -- Generate KeySource and namelist files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 0.5          07/12/93        Initial.
 ***************************************************************************/

#include "global.h"
#include <ctype.h>

#define BANNER "CBGEN  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V0.5 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V0.5\n");
#endif

/* structure to hold players name */
struct namenode {
    struct namenode *left, *right;     /* children */
    short initial;                     /* uppercase initial letter */
    char name[1];                      /* name (allocated size) */
};

typedef struct namenode *NameNode;

static int quiet = 0;                  /* -q flag */
static u_long first = 1L, last = 0xffffffffL;


/* infomation type */

#define T_PLAYERS       1

/* source/target */

#define NAMELIST        1
#define DATABASE        2
#define KEYSOURCE       3

static int type = 0, source = 0, target = 0;

static struct namenode nameroot = {NULL, NULL, 0, "M"};

static void usage P__((void));
static int p_ip_nl P__((FILE * ifp));
static int p_ip_db P__((Database db));
static int p_op_ks P__((FILE * ofp));
static int p_op_nl P__((FILE * ofp));
static void add_namenode P__((NameNode node, NameNode new));
static void free_namenode P__((NameNode node));
static void output_name P__((NameNode node, int initial, FILE * file, char *format));
static char *getline P__((FILE * file));
static FILE *open_strfile P__((char *name, char *mode));
static NameNode alloc_namenode P__((char *name));
static int process_name P__((char *name));

/* the format strings used to output White and Black Player keys */
#define WHITEPLAYER "\t\tkey \"%s\" {\n\t\t\tplayer \"%s.*-\"\n\t\t}\n"
#define BLACKPLAYER "\t\tkey \"%s\" {\n\t\t\tplayer \"-%s\"\n\t\t}\n"

static void
usage()
{
    error("usage: cbgen [options] operation input-file output-file");
    error("options:");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify first and last game to generate from");
    error("operation = <type><source><target>");
    error("type:");
    error("  p:\tplayers names");
    error("source:");
    error("  n:\tnamelist");
    error("  d:\tdatabase");
    error("target:");
    error("  n:\tnamelist");
    error("  k:\tKeySource");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret, status;
    char *op, *ipfile, *opfile;
    FILE *ifp, *ofp;
    Database db;

    opterr = 0;
    while ((c = getopt(argc, argv, "qr:")) != EOF) {
        switch (c) {
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (argc != 3)
        usage();

    ret = 0;

    op = *argv++;

    if (*op == 'p')
        type = T_PLAYERS;
    op++;

    if (*op == 'n')
        source = NAMELIST;
    else if (*op == 'd')
        source = DATABASE;
    op++;

    if (*op == 'n')
        target = NAMELIST;
    else if (*op == 'k')
        target = KEYSOURCE;

    if (!type) {
        error("unknown information type");
        return 1;
    }
    if (!source) {
        error("information source not specified");
        return 1;
    }
    if (!target) {
        error("information target not specified");
        return 1;
    }
    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    ipfile = *argv++;
    opfile = *argv;

    if (source == NAMELIST) {
        ifp = open_strfile(ipfile, "rt");
        if (!ifp)
            return 1;
    } else if (source == DATABASE) {
        db = open_database(ipfile);
        if (!db)
            return 1;
        if (last > db->ngames)
            last = db->ngames;
        if (first > last)
            first = last;
    }
    if (target == KEYSOURCE || target == NAMELIST) {
        ofp = open_strfile(opfile, "wt");
        if (!ofp) {
            ret = 1;
            goto quit;
        }
    }
    if (type == T_PLAYERS) {
        switch (source) {
        case NAMELIST:
            status = p_ip_nl(ifp);
            break;
        case DATABASE:
            status = p_ip_db(db);
            break;
        default:
            status = -1;
        }

        if (status == 0) {
            switch (target) {
            case NAMELIST:
                status = p_op_nl(ofp);
                break;
            case KEYSOURCE:
                status = p_op_ks(ofp);
                break;
            default:
                status = -1;
            }

            free_namenode(&nameroot);

            if (status < 0)
                ret = 1;
        }
    }
quit:
    if (source == NAMELIST)
        fclose(ifp);
    else if (source == DATABASE)
        close_database(db);
    if (target == NAMELIST || target == KEYSOURCE)
        fclose(ofp);
    return ret;
}

/*
 * P_IP_NL
 *
 * Get players names from a namelist.
 */
static int
p_ip_nl(ifp)
    FILE *ifp;
{
    char *name;
    NameNode new;
    u_long i;

    if (!quiet)
        output("reading namelist\n");

    i = 0L;
    while (name = getline(ifp)) {
        if (!quiet) {
            i++;
            if ((int) i % 50 == 0)
                output("%lu\r", i);
        }
        if (process_name(name))
            continue;
        new = alloc_namenode(name);
        if (!new) {
            return -1;
        }
        add_namenode(&nameroot, new);
    }
    return 0;
}

/*
 * P_IP_DB
 *
 * Input players names from a database
 */
static int
p_ip_db(db)
    Database db;
{
    register u_long i;
    int ret;
    Game game;
    NameNode new;
    u_char *cptr;

    ret = 0;

    if (!quiet)
        output("reading database\n");

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game)
        return -1;

    if (!quiet)
        output("(%lu / %lu)\r", first, last);

    for (i = first; i <= last; i++) {

        if (!quiet) {
            if ((int) i % 50 == 0)
                output("(%lu / %lu)\r", i, last);
        }
        game_tidy(game);
        game->num = i;
        if (read_info(db, game) < 0)
            continue;

        if (!game->plen)
            continue;

        cptr = (u_char *) index((char *) game->pinfo, '-');
        if (cptr)
            *cptr = '\0';

        if (process_name((char *) game->pinfo))
            continue;
        new = alloc_namenode((char *) game->pinfo);
        if (!new) {
            ret = -1;
            break;
        }
        add_namenode(&nameroot, new);

        if (cptr && *(cptr + 1)) {
            if (process_name((char *) cptr + 1))
                continue;
            new = alloc_namenode((char *) cptr + 1);
            if (!new) {
                ret = -1;
                break;
            }
            add_namenode(&nameroot, new);
        }
    }

    if (!quiet && ret == 0 && i > last)
        output("(%lu / %lu)\r", last, last);

    free(game);
    return ret;
}

/*
 * P_OP_KS
 *
 * Output the players names to a KeySource file.
 */
static int
p_op_ks(ofp)
    FILE *ofp;
{
    int initial;

    if (!quiet)
        output("writing KeySource\n");

    fprintf(ofp, "#\n");
    fprintf(ofp, "# Automatically generated by KSGEN (c)1993 Andy Duplain.\n");
    fprintf(ofp, "#\n");
    fprintf(ofp, "subkey_related\ttrue\n");
    fprintf(ofp, "info_tests_only\ttrue\n");

 /* output white name keys */

    fprintf(ofp, "key \"White names\" {\n");
    for (initial = 'A'; initial <= 'Z'; initial++) {

        if (!quiet)
            output("White names %c...\r", initial);

        fprintf(ofp, "\tkey \"%c...\" {\n", initial);
        fprintf(ofp, "\t\tplayer \"^[%c%c].*-\"\n", tolower(initial),
          initial);
        output_name(&nameroot, initial, ofp, WHITEPLAYER);
        fprintf(ofp, "\t}\n");
    }
    fprintf(ofp, "}\n");

    fprintf(ofp, "key \"Black names\" {\n");
    for (initial = 'A'; initial <= 'Z'; initial++) {

        if (!quiet)
            output("Black names %c...\r", initial);

        fprintf(ofp, "\tkey \"%c...\" {\n", initial);
        fprintf(ofp, "\t\tplayer \"-[%c%c]\"\n", tolower(initial),
          initial);
        output_name(&nameroot, initial, ofp, BLACKPLAYER);
        fprintf(ofp, "\t}\n");
    }
    fprintf(ofp, "}\n");

    return 0;
}

/*
 * P_OP_NL
 *
 * Output the players names to a namelist.
 */
static int
p_op_nl(ofp)
    FILE *ofp;
{
    int initial;

    if (!quiet)
        output("writing namelist\n");

    for (initial = 'A'; initial <= 'Z'; initial++) {

        if (!quiet)
            output("%c...\r", initial);

        output_name(&nameroot, initial, ofp, "%s\n");
    }

    return 0;
}


/*
 * ADD_NAMENODE
 *
 * Add a name to the binary tree
 */
static void
add_namenode(node, new)
    NameNode node, new;
{
    int res;

    res = strcmp(node->name, new->name);
    if (res > 0) {
        if (node->left)
            add_namenode(node->left, new);
        else
            node->left = new;
    } else if (res < 0) {
        if (node->right)
            add_namenode(node->right, new);
        else
            node->right = new;
    } else {                           /* res == 0 */
        free(new);
    }
}

/*
 * FREE_NAMENODE
 *
 * Free a node and it's children
 */
static void
free_namenode(node)
    NameNode node;
{
    if (node->left)
        free_namenode(node->left);
    if (node->right)
        free_namenode(node->right);
    if (node->initial) {               /* not the root node */
        free(node);
    }
}


/*
 * OUTPUT_NAME
 *
 * Output all names in the tree whose initial letter matches that passed
 * in.
 */
static void
output_name(node, initial, file, format)
    NameNode node;
    int initial;
    FILE *file;
    char *format;
{
    if (node->left)
        output_name(node->left, initial, file, format);
    if (initial == node->initial)
        fprintf(file, format, node->name, node->name);
    if (node->right)
        output_name(node->right, initial, file, format);
}

/*
 * GETLINE
 *
 * Get a line from the input file and remove the trailing newline
 */
static char *
getline(file)
    FILE *file;
{
    static char line[80];
    int len;

    if (!fgets(line, sizeof(line), file))
        return NULL;

    len = strlen(line);
    if (len && line[len - 1] == '\n') {
        line[len - 1] = '\0';
    }
    return line;
}

/*
 * OPEN_STRFILE
 *
 * Open a streams file and print an error message upon error.
 */
static FILE *
open_strfile(name, mode)
    char *name, *mode;
{
    FILE *ret;

    ret = fopen(name, mode);
    if (!ret) {
        error("error opening file \"%s\"\n", name);
        return NULL;
    }
    return ret;
}

/*
 * ALLOC_NAMENODE
 *
 * Allocate memory for struct name node and initialise with name.
 */
static NameNode
alloc_namenode(name)
    char *name;
{
    NameNode new;

    new = (NameNode) mem_alloc(sizeof(struct namenode) + strlen(name) + 1);
    if (!new) {
        return NULL;
    }
    new->initial = (int) toupper(*name);
    strcpy(new->name, name);

    return new;
}

/*
 * PROCESS_NAME
 *
 * Remove any trailing spaces and other garbage.
 *
 * Returns 1 if names completely removed after processing.
 */
static int
process_name(name)
    char *name;
{
    register char *cptr;

    cptr = index(name, '(');
    if (cptr)
        *cptr = '\0';

    cptr = index(name, ';');
    if (cptr)
        *cptr = '\0';


    cptr = name + (strlen(name) - 1);

    while (cptr >= name) {
        if (*cptr == ' ' || *cptr == '\t')
            *cptr = '\0';
        else
            break;
        cptr--;
    }

    return *name ? 0 : 1;
}
