/***************************************************************************
 * CBFBK - Generate a Fritz Opening Book from a ChessBase database.
 *
 * Copyright (c)1994 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          17/08/94        Initial.
 * 1.1			16/09/94		Added support for '!?' and '?!' for Fritz3,
 *								and search for transpositions.
 ***************************************************************************/

#include "global.h"
#ifdef MSDOS
#include <io.h>
#endif

#define BANNER "CBFBK  Copyright (c)1994 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.1 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.1\n");
#endif

struct fbkmove {
	u_char from, to;	/* from and to square */
	u_short flags;		/* move flags and evaluation */
	struct fbkmove *next;	/* pointer to next move in line */
	struct fbkmove *next_var, *prev_var;	/* pointers variation moves */
};
typedef struct fbkmove *FBKmove;

/* fbkmove.flags defines */

#define W_EP		0x8000		/* En Passant (for white) */
#define B_EP		0x4000		/* En Passant (for black) */
#define W_KS		0x2000		/* Kingside castle (for white) */
#define W_QS		0x1000		/* Queenside castle (for white) */
#define B_KS		0x0800		/* Kingside castle (for black) */
#define B_QS		0x0400		/* Queenside castle (for black) */
#define TRANS		0x0200		/* 'next' points to transposition */
#define EVAL_MASK	0x0003		/* Evaluation mask */
#define EVAL_NONE	0x0000		/* No evaluation */
#define EVAL_BAD	0x0001		/* ? */
#define EVAL_INT	0x0002		/* !? */
#define EVAL_DUB	0x0003		/* ?! */

static void usage P__((void));
static int init_book P__((void));
static void new_book_line P__((void));
static int add_book P__((int halfmove));
static FBKmove add_lastmove P__((int halfmove, u_char eval));
static FBKmove find_lastmove P__((FBKmove node));
static int find_position P__((FBKmove node, int halfmove));
static int test_position P__((FBKmove node, int halfmove));
static u_char get_eval P__((void));
static void write_line P__((FBKmove node));
static void write_move P__((FBKmove node, int var));
static u_char convert_offset P__((int cbbyte));
static void free_book P__((FBKmove node));
static void output_lastmove P__((char *msg));
static int init_pos_stack P__((void));
static int free_pos_stack P__((void));
static int push_position P__((void));
static int pop_position P__((void));
static int peek_position P__((void));

#define DEF_MAX_MOVE	10
static int max_move = DEF_MAX_MOVE;		/* -m argument */
static int quiet = 0;					/* -q flag */
static int verbose = 0;					/* -v flag */
static int transpositions = 1;			/* -t flag */
static int elim = 1;					/* -e flag */
static char *dbname, *fbkname = NULL;
static FILE *fbkfile = NULL;
static u_long first = 1L, last = 0xffffffffL;
static Database db = NULL;
static Game game = NULL;
struct fbkmove root; 
static FBKmove mptr;
static u_char *comptr;
static u_char position[64];		/* current position being looked for */
static u_char *pos_stack = NULL;	/* position stack */
static int stack_idx = 0;		/* index of current position in stack */
static u_char *stack_ptr;
static int stack_err = 0;		/* position-stack error indicator */
#define NPOS_STACK	511			/* number of positions in stack */
static FBKmove current_node, found_position;	/* transpositions */
static int find_colour;			/* colour to move, to find */
static u_char *rep_list;		/* list of positions that have occurred in
								   this game */
static u_char *rep_ptr;			/* pointer to position in rep_list */
#define NREP_LIST 511			/* number of positions in rep_list */
static int nrep_list = NREP_LIST;

static void
usage()
{
	error("usage: cbfbk [options] database");
	error("options:");
	error("  -q\t\tquiet");
	error("  -r x-y\tspecify the first and last game to use");
	error("  -m x\t\tspecify the maximum move number for each line (default %s)",
	    DEF_MAX_MOVE);
	error("  -t\t\tdon't search for transpositions");
	error("  -e\t\tdon't eliminate transpositions in disk book (Fritz1 & 2)");
	error("  -v\t\tverbose");
	error("\nThe Fritz opening book with be generated as <database>.fbk");
	exit(1);
}

int
main(argc, argv)
int argc;
char **argv;
{
	int c, ret, last_report, progress;
	u_long i, ngames;

	ret = 0;
    last_report = -1;

	opterr = 0;
	while ((c = getopt(argc, argv, "qr:em:tv")) != EOF) {
		switch (c) {
		case 'q':
			quiet++;
			verbose--;
			break;
		case 'r':
			first = range_first(optarg);
			last = range_last(optarg);
			if ((!first || !last) || first > last) {
				error("invalid range");
				return 1;
			}
			break;
		case 'e':
			elim = 0;
			break;
		case 'm':
			max_move = (int)xatol(optarg);
			if (xatol_err) {
				error("invalid move number (use \"-m 0\" for unlimited moves)");
				return 1;
			}
			break;
		case 't':
			transpositions = 0;
			break;
		case 'v':
			verbose++;
			quiet--;
			break;
		case '?':
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc != 1)
		usage();

	dbname = *argv;
	kill_ext(dbname);

	if (!quiet) {
		output(BANNER);
		VERSION();
	}
	db = open_database(dbname);
	if (!db)
		return 1;

	if (last > db->ngames)
		last = db->ngames;
	if (first > last)
		first = last;

	fbkname = derive_name(dbname, ".fbk");
	if (!fbkname) {
		ret = -1;
		goto quit;
	}
	fbkfile = fopen(fbkname, "w+b");
	if (!fbkfile) {
		error("Couldn't create book file \"%s\"", fbkname);
		ret = -1;
		goto quit;
	}

	game = (Game) mem_alloc(sizeof(struct game));
	if (!game) {
		ret = -1;
		goto quit;
	}

	if (init_book() < 0) {
		ret = -1;
		goto quit;
	}

	if (init_pos_stack() < 0) {
		ret = -1;
		goto quit;
	}

	if (max_move)
		nrep_list = (max_move * 2) + 2;
	rep_list = (u_char *)mem_alloc(nrep_list * 64);
	if (!rep_list) {
		ret = -1;
		goto quit;
	}

    ngames = (last - first) + 1L;
	mem_alloc_err = 0;

	/* process all the games in the requested range */
	for (i = first; i <= last; i++) {
		game->num = i;
		if (read_game(db, game) < 0)
			continue;
		if (is_partial(game->header) || is_deleted(game->header))
			continue;
		new_book_line();
		if (verbose)
			output("[%ld/%ld]\n", i, last);
		if (process_moves(game, add_book) < 0) {
			/* if mem_alloc() error, then write what we have.. */
			if (mem_alloc_err)
				break;
			ret = -1;
			goto quit;
		}
		if (stack_err) {
			ret = -1;
			goto quit;
		}
		if (verbose)
			newline();
		game_tidy(game);
        if (!quiet && !verbose) {
            progress = (int) ((i * 100L) / ngames);
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
	}

	write_line(root.next);

quit:
	if (rep_list)
		free(rep_list);
	free_pos_stack();
	free_book(root.next);
	if (game)
		free(game);
	if (fbkname)
		free(fbkname);
	if (fbkfile)
		fclose(fbkfile);
	close_database(db);
	if (!ret && !quiet) {
		output("Don't forget to copy the classifcation keys for use by Fritz:\n");
		output("\t<database>.cpo -> <database>.cpb\n");
		output("\t<database>.cko -> <database>.ckb\n");
		output("and then copy the .fbk, .cpb and .ckb into the Fritz books directory\n");
	}

	return ret;
}

/*
 * INIT_BOOK
 *
 * Initialise the internal book representation
 */
static int
init_book()
{
	root.from = 0;
	root.to = 0;
	root.flags = 0;
	root.next = NULL;
	root.next_var = NULL;
	root.prev_var = NULL;
	new_book_line();
	return 0;
}

/*
 * NEW_BOOK_LINE
 *
 * Initialise for a fresh game (new line).
 */
static void
new_book_line()
{
	mptr = &root;
	rep_ptr = rep_list;
	if (game->clen)
		comptr = game->comments;
	else
		comptr = NULL;
}

/*
 * ADD_BOOK
 *
 * Add a move to the internal book representation.
 */
static int
add_book(halfmove)
	int halfmove;
{
	int no_test, i;
	u_char eval;
	u_char *ptr;

	if (halfmove == START_VAR_SIGNAL)
		return 0;
	else if (halfmove == END_VAR_SIGNAL)
		return 0;

	eval = get_eval(); 		/* examine evaluations, regardless of whether
							   we will use the move */
	if (var_level)
		return 0;			/* variations not supported */

	if (max_move && to_move(halfmove) > max_move) {
			if (verbose)
				output("<max>");
			return 1;		/* finished with this game */
	}	

	if (lastmove.prom) {
		if (verbose)
			output("<promotion>");
		return 1;			/* cannot support promotions; so no more from
							   this game */
	}	

	/* avoid a nasty problem that can occur when a repetition happens
	   in the game (the transposition code will find the repetition and
	   create a loop in the tree which upsets the other tree functions) */

	if (halfmove >= nrep_list) {
		if (verbose)
			output("<repetition-list full>");
		return 1;
	}
	copy_board(cb_board, rep_ptr);		/* first copy the current position */
	rep_ptr += 64;
	for (i = 0, ptr = rep_list; i < halfmove - 1; i++, ptr += 64) {
		if (compare_board(ptr, cb_board)) {
			if (verbose)
				output("<repetition>");
			return 1;
		}
	}

	no_test = 0;
	if (!mptr->next) {
		/* add new move to the current line */
		mptr->next = add_lastmove(halfmove, eval);
		if (!mptr->next)
			return -1;
		mptr = mptr->next;
		output_lastmove("add:");
	} else {
		/* following a line */
		FBKmove next = mptr->next;
		mptr = find_lastmove(next);
		if (!mptr) {
			/* a new variation */
			while (next->next_var)
				next = next->next_var;
			next->next_var = add_lastmove(halfmove, eval);
			if (!next->next_var)
				return -1;
			mptr = next->next_var;
			mptr->prev_var = next;	/* link back */
			output_lastmove("var:");
		} else {
			/* following an existing line */
			if (eval != EVAL_NONE && (mptr->flags & EVAL_MASK) == EVAL_NONE)
				mptr->flags |= eval & EVAL_MASK;	/* set eval if none */
			no_test++;
			output_lastmove(NULL);
		}
	}

	if (transpositions && !no_test) {
		found_position = NULL;
		current_node = mptr;
		find_colour = to_colour(halfmove);
		init_board(position);
  		find_position(root.next, 1);
		if (found_position) {
			mptr->next = found_position->next;
			if (mptr->next)
				mptr->flags |= TRANS;
			if (verbose)
				output("<trans> ");
		}
	}

	return 0;
}

/*
 * FIND_LASTMOVE
 *
 * Find the lastmove at the current level (i.e. this move and all the other
 * variations).
 */
static FBKmove
find_lastmove(node)
	FBKmove node;
{
	FBKmove saved;

	if (!node)
		return NULL;

	saved = node;

	/* search this node and 'next' variations */
	while (node) {
		if (node->from == lastmove.from && node->to == lastmove.to)
			return node;
		node = node->next_var;
	}

	/* search 'previous' variations */
	node = saved->prev_var;
	while (node) {
		if (node->from == lastmove.from && node->to == lastmove.to)
			return node;
		node = node->prev_var;
	}

	return NULL;
}

/*
 * ADD_LASTMOVE
 *
 * Allocate memory for another node and copy in the details from the
 * lastmove.
 */
static FBKmove
add_lastmove(halfmove, eval)
	int halfmove;
	u_char eval;
{
	FBKmove ret;

	ret = (FBKmove)mem_alloc(sizeof(struct fbkmove));
	if (!ret)
		return NULL;
	ret->from = lastmove.from;
	ret->to = lastmove.to;
	ret->flags = eval;

	/* set move flags so that find_position() can perform moves more
	   easily (En Passant and Castling) */
	if (to_colour(halfmove)) {
		/* black */
		if (lastmove_flags & ENPASSANT)
			ret->flags |= B_EP;
		else if (lastmove_flags & SHORT_CASTLE)
			ret->flags |= B_KS;
		else if (lastmove_flags & LONG_CASTLE)
			ret->flags |= B_QS;
	} else {
		/* white */
		if (lastmove_flags & ENPASSANT)
			ret->flags |= W_EP;
		else if (lastmove_flags & SHORT_CASTLE)
			ret->flags |= W_KS;
		else if (lastmove_flags & LONG_CASTLE)
			ret->flags |= W_QS;
	}
	return ret;
}

/*
 * FIND_POSITION
 *
 * Find a position in the tree.
 *
 */
static int
find_position(node, halfmove)
	FBKmove node;
	int halfmove;
{
	FBKmove vnode;

	while (node) {

		if (push_position() < 0)
			return -1;

		/* search 'next' variations */
		vnode = node->next_var;
		while (vnode) {
			if (peek_position() < 0)
				return -1;
			if (test_position(vnode, halfmove))
				goto found;
			if (find_position(vnode->next, halfmove+1))
				goto found;
			vnode = vnode->next_var;
		}

		/* search 'previous' variations */
		vnode = node->prev_var;
		while (vnode) {
			if (peek_position() < 0)
				return -1;
			if (test_position(vnode, halfmove))
				goto found;
			if (find_position(vnode->next, halfmove+1))
				goto found;
			vnode = vnode->prev_var;
		}

		if (pop_position() < 0)
			return -1;

		if (test_position(node, halfmove))
			return 1;
			
		node = node->next;
		halfmove++;
	}

	return 0;

found:
	if (pop_position() < 0)
		return -1;
	return 1;
}

/*
 * TEST_POSITION
 *
 * Apply a move to a position and test it against the current position.
 */
static int
test_position(node, halfmove)
	FBKmove node;
	int halfmove;
{
	register u_short flags;

	position[node->to] = position[node->from];
	position[node->from] = 0;
	flags = node->flags;
	if (flags & W_EP) {			/* white captures en passant */
		position[node->to - 1] = 0;
	} else if (flags & B_EP) {	/* black captures en passant */
		position[node->to + 1] = 0;
	} else if (flags & W_KS) {	/* white castles kingside */
		position[56] = 0;
		position[40] = ROOK;
	} else if (flags & W_QS) {	/* white castles queenside */
		position[0] = 0;
		position[16] = ROOK;
	} else if (flags & B_KS) {	/* black castles kingside */
		position[63] = 0;
		position[47] = ROOK | 8;
	} else if (flags & B_QS) {	/* black castles queenside */
		position[7] = 0;
		position[23] = ROOK | 8;
	}
	if (to_colour(halfmove) == find_colour &&
		current_node != node) {
		if (compare_board(position, cb_board)) {
			found_position = node;
			return 1;
		}
	}
	return 0;
}

/*
 * GET_EVAL
 *
 * Determines if the current move has a evaluation of '?', '?!' or '!?'
 */
static u_char
get_eval()
{
	u_char eval;
	u_char ret;

	if (!comptr)
		return 0;	/* no comments in this game */

	if (!comment || *comptr != 0xff)
		return 0;	/* no comment for this move */

	eval = 0;
	if (*++comptr != 0xff) {	/* evaluation */
		eval = *comptr;
		if (*++comptr != 0xff) {	/* position evaluation */
			/* poseval = *comptr; */
			if (*++comptr != 0xff) {	/* move evaluation */
				/* moveeval = *comptr; */
				if (*++comptr != 0xff) {	/* null */
					if (*++comptr != 0xff) { /* annotation */
						while (*++comptr != 0xff)
							;
					}
				}
			}
		}
	}

	switch (eval) {
	case 2:		/* ? */
	case 6:		/* ?? */
		ret = EVAL_BAD;
		break;
	case 3:		/* !? */
		ret = EVAL_INT;
		break;
	case 4:		/* ?! */
		ret = EVAL_DUB;
		break;
	default:
		ret = EVAL_NONE;
		break;
	}
	return ret;
}

#define VARSTART	0x40		/* bit clear if start of variation */
#define VAREND		0x80		/* bit set if end of variation */
#define BADMOVE		0x80
#define INTMOVE		0x40
#define DUBMOVE		0xc0

/*
 * WRITE_LINE
 *
 * Write a line to the disk book.  If we are eliminating transpositions from
 * the disk book and the next move is a transposition then we don't write
 * it out; this way Fritz3 will detect the transpositons itself.
 */
static void
write_line(node)
	FBKmove node;
{
	FBKmove vnode;

	while (node) {

		/* write 'next' variations */
		vnode = node->next_var;
		while (vnode) {
			write_move(vnode, 1);
			if (!(elim && (vnode->flags & TRANS)))
				write_line(vnode->next);
			vnode = vnode->next_var;
		}

		/* write 'previous' variations */
		vnode = node->prev_var;
		while (vnode) {
			write_move(vnode, 1);
			if (!(elim && (vnode->flags & TRANS)))
				write_line(vnode->next);
			vnode = vnode->prev_var;
		}

		write_move(node, 0);
		if (elim && (node->flags & TRANS))
			break;
		node = node->next;
	}
}

/*
 * WRITE_MOVE
 *
 * Write a move to the disk book.
 */
static void
write_move(node, var)
	FBKmove node;
	int var;
{
	u_char data[2];

	data[0] = convert_offset(node->from);
	data[1] = convert_offset(node->to);

	if (!var)
		data[0] |= VARSTART;		/* reverse logic */
	if (!node->next || (elim && (node->flags & TRANS)))
			data[0] |= VAREND;

	switch(node->flags & EVAL_MASK) {
	case EVAL_BAD:
		data[1] |= BADMOVE;
		break;
	case EVAL_INT:
		data[1] |= INTMOVE;
		break;
	case EVAL_DUB:
		data[1] |= DUBMOVE;
		break;
	case EVAL_NONE:
	default:
		break;
	}

	fwrite((char *)data, 1, 2, fbkfile);
}

/*
 * CONVERT_OFFSET
 *
 * Convert the ChessBase-style board offset byte to the Fritz2 style.
 */
static u_char
convert_offset(cbbyte)
	int cbbyte;
{
	return (to_rank(cbbyte) << 3) | to_file(cbbyte);
}

/*
 * FREE_BOOK
 *
 * Free the book nodes.
 */
static void
free_book(node)
	FBKmove node;
{
	FBKmove next, vnode;
	while (node) {
		vnode = node->next_var;
		while (vnode) {
			if (!(vnode->flags & TRANS))
				free_book(vnode->next);
			next = vnode->next_var;
			free(vnode);
			vnode = next;
		}
		vnode = node->prev_var;
		while (vnode) {
			if (!(vnode->flags & TRANS))
				free_book(vnode->next);
			next = vnode->prev_var;
			free(vnode);
			vnode = next;
		}
		if (node->flags & TRANS)
			next = NULL;		/* next node is a transposition; stop now */
		else 
			next = node->next;
		free(node);
		node = next;
	}
}

/* 
 * OUTPUT_LASTMOVE
 *
 * Output the status of the lastmove.
 */
static void
output_lastmove(msg)
	char *msg;
{
	if (!verbose)
		return;

 	output("%s%s ", msg ? msg : "",
	  format_move(lastmove.from, lastmove.to, lastmove.prom));
}

/*
 * INIT_POS_STACK
 *
 * Initilise the position stack.
 */
static int
init_pos_stack()
{
	if (pos_stack)
		return 0;
	pos_stack = (u_char *)mem_alloc(64 * NPOS_STACK);
	if (!pos_stack)
		return -1;
	stack_idx = 0;
	stack_ptr = pos_stack;
	stack_err = 0;
	return 0;
}

/*
 * FREE_POS_STACK
 *
 * Free resources used by the position stack.
 */
static int
free_pos_stack()
{
	if (!pos_stack)
		return 0;
	free(pos_stack);
	pos_stack = 0;
	return 0;
}

/*
 * PUSH_POSITION
 *
 * Push the current position (in 'position') onto the position stack.
 */
static int
push_position()
{
	if (stack_idx >= NPOS_STACK) {
		error("position-stack overflow");
		stack_err++;
		return -1;
	}
	copy_board(position, stack_ptr);
	stack_ptr += 64;
	stack_idx++;
	return 0;
}

/*
 * POP_POSITION
 *
 * Make the position on the top of the stack the current position.
 */
static int
pop_position()
{
	if (stack_idx <= 0) {
		error("position-stack underflow");
		stack_err++;
		return -1;
	}
	stack_ptr -= 64;
	stack_idx--;
	copy_board(stack_ptr, position);
	return 0;
}

/*
 * PEEK_POSITION
 *
 * Make the position on the top of the stack the current position without
 * removing it from the stack.
 */
static int
peek_position()
{
	if (stack_idx <= 0) {
		error("position-stack underflow");
		stack_err++;
		return -1;
	}
	copy_board(stack_ptr - 64, position);
	return 0;
}
