/***************************************************************************
 * CBEPD - Convert between EPD and a ChessBase database
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          18/08/94        Initial.
 ***************************************************************************/

#include "global.h"
#ifdef MSDOS
#include <io.h>
#endif

#include <ctype.h>
#include <time.h>

#define BANNER "CBEPD  Copyright (c)1994 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.0 [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.0\n");
#endif

static void usage P__((void));
static int export P__((void));
static int export_move P__((int halfmove));
static int output_pos P__((void));
static char *find_annot P__((void));
static int fritz_ce P__((char **cpptr));
static int fritz_moves P__((char **cpptr));
static int fritz_acs P__((char **cpptr));
static int import P__((void));
static int read_epd P__((void));
static char *parse_pos P__((void));
static int dispatch_op P__((char *opname, int (*func)(char *opstr)));
static int process_moveop P__((char *opstr));
static int process_fmvn P__((char *opstr));
static int process_id P__((char *opstr));
static int add_move P__((u_char move));

static int quiet = 0;                  /* -q flag */
static char *dbname, *epdname;
static u_long first = 1L, last = 0xffffffffL;
static Database db = NULL;
static Game game = NULL;
static int conv = 0;					/* -e/-i flags */
static int append = 0;					/* -a flag */
static int last_only = 0;				/* -l flag */
static int ignore_fritz = 0;			/* -I flag */
static FILE *file;

#define EXPORT	1
#define IMPORT	2

static void
usage()
{
    error("usage: cbepd -i|-e [options] epd-file database");
    error("general options:");
	error("  -e\t\texport (ChessBase -> EPD)");
	error("  -i\t\timport (EPD -> ChessBase)");
    error("  -q\t\tquiet");
	error("\nexport-only options:");
    error("  -a\t\topen epd-file in \"append\" mode");
	error("  -l\t\tonly convert last game in database");
	error("  -I\t\tIgnore Fritz analysis annotations");
    error("  -r x-y\tspecify the first and last game to convert");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, ret;
	char *mode;

    ret = 0;

    opterr = 0;
    while ((c = getopt(argc, argv, "qr:eialI")) != EOF) {
        switch (c) {
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
			break;
        case 'e':
            if (conv) {
                error("specify either \"-e\" or \"-i\"");
                return 1;
            }
            conv = EXPORT;
            break;
        case 'i':
            if (conv) {
                error("specify either \"-e\" or \"-i\"");
                return 1;
            }
            conv = IMPORT;
            break;
		case 'a':
			append++;
			break;
		case 'l':
			last_only++;
			break;
		case 'I':
			ignore_fritz++;
			break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (!conv) {
        error("you must specify \"-e\" to export or \"-i\" to import");
        return 1;
    }

    if (argc != 2)
        usage();

	epdname = *argv++;
    dbname = *argv;
    kill_ext(dbname);

    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    no_error = 1;
    db = open_database(dbname);
    no_error = 0;
    if (!db) {
        if (conv == IMPORT) {
            if (!quiet)
                output("creating database %s\n", dbname);
            db = create_database(dbname);
            if (!db)
                return 1;
        } else {                       /* exporting */
            error("error opening database %s", dbname);
            return 1;
        }
    }
    if (conv == EXPORT) {
		if (last_only) {
			first = last = db->ngames;
		} else {
        	if (last > db->ngames)
            	last = db->ngames;
        	if (first > last)
            	first = last;
		}
    }

	/* open the EPD file */
    if (conv == IMPORT)
        mode = "rt";
    else
        mode = append ? "at" : "wt";
    file = fopen(epdname, mode);
    if (!file) {
        error("error opening \"%s\"", epdname);
        return 1;
    }
    if (conv == EXPORT) {
        ret = export();
    } else {                           /* importing */
        ret = import();
    }

    close_database(db);
    fclose(file);

    return ret == -1 ? 1 : 0;
}

static char acs_str[8];				/* acs opcode string */
static char ce_str[8];				/* ce opcode string */
static char dm_str[8];				/* dm opcode string */
static char fmvn_str[8];			/* fmvn opcode string */
static char id_str[256];			/* id opcode string */
static char pm_str[8];				/* pm opcode string */
static char pv_str[512];			/* pv opcode string */
static char sm_str[8];				/* sm opcode string */

struct opstrp {
	char *name, *str;
};

static struct opstrp opstrs[] = {
	{ "acs", acs_str },
	{ "ce", ce_str },
	{ "dm", dm_str },
	{ "fmvn", fmvn_str },
	{ "id", id_str },
	{ "pm", pm_str },
	{ "pv", pv_str },
	{ "sm", sm_str }
};

#define NOPSTRS (sizeof(opstrs) / sizeof(opstrs[0]))

/*
 * EXPORT
 *
 * Convert a ChessBase file to EPD
 */
static int
export()
{
	u_long i;
	int j;

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game)
        return 1;

    for (i = first; i <= last; i++) {

		/* initialise opcode strings */
		for (j = 0; j < NOPSTRS; j++)
			*opstrs[j].str = '\0';

		/* read game */
        game->num = i;
        if (read_game(db, game) < 0)
            continue;

		/* output position information */
		output_pos();

		/* set fmvn and id strings from game header */
		if (to_move(game->halfmove) > 1)
			sprintf(fmvn_str, "%d", to_move(game->halfmove));
		if (strncmp((char *)game->pinfo, "ID: ", 4) == 0)
			sprintf(id_str, "%s", (char *)&game->pinfo[4]);

		/* generate move and comment-related information */
        process_moves(game, export_move);

		/* only allow a 'sm' if it's different from the 'pm' opcode */
		if (sm_str[0]) {
			int len;
			len = strlen(sm_str);
			if (strncmp(sm_str, pm_str, len) == 0)
				sm_str[0] = '\0';
		}

		/* output non-empty opcode strings */
		for (j = 0; j < NOPSTRS; j++)
			if (*opstrs[j].str)
				fprintf(file, "%s %s; ", opstrs[j].name, opstrs[j].str);

        game_tidy(game);
		if (!quiet)
			output("game: %lu\r", i);
		fputs("\n", file);			/* terminate line with newline */
    }

    free(game);
    return 0;
}

/*
 * EXPORT_MOVE
 *
 * Output 'predicted variation' or 'predicted move'
 */
static int
export_move(halfmove)
	int halfmove;
{
	int len, dummy;
	char *cptr;

	if (halfmove == START_VAR_SIGNAL || halfmove == END_VAR_SIGNAL
		|| var_level)
		return 0;		/* not interested in variations */

	if (!ignore_fritz && halfmove == (int)game->halfmove &&
	    (cptr = find_annot())) {
		/* if this is the first move and it contains a Fritz comment, then
		   generate all information from it */

		/* process Fritz evaluation */
	   	if (fritz_ce(&cptr) < 0)
			goto invalid_annot;

		/* ignore Fritz depth indication */
	   	cptr = get_word(cptr, &len);
	   	if (!cptr)
			goto invalid_annot;
		if (sscanf(cptr, "(%d/%d):", &dummy, &dummy) != 2)
				goto invalid_annot;
		cptr += len;

		/******************************************************************
		 * After fritz_moves() is called it is no longer possible to goto *
		 * invalid_annot, as the board, and other move-related values     *
		 * have been reset and procmove() [the function that calls this	  *
		 * function] cannot continue.									  *
		 ******************************************************************/

		/* process Fritz variation */
	   	if (fritz_moves(&cptr) < 0)
			return -1;

		/* process Fritz analysis time */
	   	if (fritz_acs(&cptr) < 0)
	   		return -1;

		return 1;		/* don't call us again; we've got all we need */

	}

invalid_annot:

	cptr = algebraic(to_colour(halfmove));

	if (halfmove == (int)game->halfmove)
		strcpy(pm_str, cptr);
	strcat(pv_str, cptr);
	strcat(pv_str, " ");
	return 0;
}

/*
 * OUTPUT_POS
 *
 * Output the position and associated information.
 */
static int
output_pos()
{
	int brdrank, brdfile, nempty;
	u_char piece;

	for (brdrank = 7; brdrank >= 0; brdrank--) {     /* rank */
		nempty = 0;
    	for (brdfile = 0, nempty = 0; brdfile < 8; brdfile++) {    /* file */
			piece = game->board[to_offset(brdfile, brdrank)];
			if (!piece) {
				nempty++;
			} else {
				if (nempty) {
					fprintf(file, "%d", nempty);
					nempty = 0;
				}
				if (piece & 8)
					fputc(piece_list[piece & 7] + ('a' - 'A'), file);
				else
					fputc(piece_list[piece], file);
				}
			}
			if (nempty)
				fprintf(file, "%d", nempty);
			if (brdrank)
				fputc('/', file);
		}
        fprintf(file, " %c ", to_colour(game->halfmove) ? 'b' : 'w');
	
		if (is_partial(game->header)) {
	        nempty = 0;		/* flag if castling rights printed or not */
			if (is_wcs(game->header)) {
				fputc('K', file);
				nempty++;
			}
			if (is_wcl(game->header)) {
				fputc('Q', file);
				nempty++;
			}
			if (is_bcs(game->header)) {
				fputc('k', file);
				nempty++;
			}
			if (is_bcl(game->header)) {
				fputc('q', file);
				nempty++;
			}
			if (!nempty)
				fputc('-', file);
		} else {
			fputs("KQkq", file);
		}

		fputc(' ', file);
		if (game->ep) {
			fputc('a' + game->ep - 1, file);
			if (to_colour(game->halfmove))
				fputc('3', file);
			else
				fputc('6', file);
		} else {
			fputc('-', file);
		}

    fputc(' ', file);
	return 0;
}

/*
 * FIND_ANNOT
 *
 * Find the annotation to the first move in the game, and create a nul-
 * terminated string from it and return a pointer to it.
 */
static char *
find_annot()
{
	u_char *cptr, *cptr1;
	int annot;

	if (!comment)
		return NULL;				/* nothing */
	cptr = game->comments;
	annot = 0;

	if (*++cptr != 0xff) {			/* evaluation */
		if (*++cptr != 0xff) {		/* position evaluation */
			if (*++cptr != 0xff) {	/* move evaluation */
				if (*++cptr != 0xff) { /* null */
					if (*++cptr != 0xff) { /* annotation */
                        annot = 1;
                    }
                }
            }
        }
    }

	if (!annot)
		return 0;
	
	/* null terminate the comment */
	for (cptr1 = cptr; *cptr1 != 0xff; cptr1++)
		;
	*cptr1 = 0;

	return (char *)cptr;
}

/*
 * FRITZ_CE
 *
 * Translate Fritz2 evaluation into EPD form.
 */
static int
fritz_ce(cpptr)
	char **cpptr;
{
	char *cptr;
	int len, sign, val;

	cptr = get_word(*cpptr, &len);
	if (!cptr)
		goto fail;

	sign = 0;
	if (*cptr == '-') {
		cptr++;
		sign = 1;
	} else if (*cptr == '+') {
		cptr++;
		sign = 0;
	}

	if (*cptr == '#') {
		/* mate in 'X' moves */
		int matein = 1;

		cptr++;

		if (*cptr == ' ') {
			*cpptr = cptr;
		} else {
			matein = (int)xatol(cptr);
			if (!xatol_ptr || *xatol_ptr != ' ')
				goto fail;
			*cpptr = xatol_ptr;
		}

		if (sign) {
			val = -32767 + (2 * matein);
		} else {
			val = 32767 - ((2 * matein) - 1);
			sprintf(dm_str, "%d", matein);
		}

	} else if (isdigit(*cptr)) {
		/* normal position evaluation */

		/* take evaluation in the form XX.YY and return XXYY */
		val = 0;
		while (*cptr && *cptr != ' ') {
			if (*cptr == '.')
				cptr++;
			else if (isdigit(*cptr)) {
				val *= 10;
				val += *cptr - '0';
				cptr++;
			} else
				goto fail;
		}
		*cpptr = cptr;
		if (sign)
			val = -val;

	} else
		goto fail;

	sprintf(ce_str, "%d", val);
	return 0;

fail:
	*cpptr = NULL;
	return -1;
}

/*
 * FRITZ_MOVES
 *
 * Tranlate Fritz variation (in coordinate notation) into a 'pm', 'pv' and/or
 * 'sm' opcode string.
 */
static int
fritz_moves(cpptr)
	char **cpptr;
{
	int len, i;
	int from, to, prom;
	int halfmove;
	char *cptr, *move_txt;
    Move moveptr;

	/* output the actual move played, as it might be different to the
	   one played in the fritz analysis */
	move_txt = algebraic(to_colour(game->halfmove));
	strcpy(sm_str, move_txt);

	/* re-initialise the board */
    copy_board(game->board, cb_board);
    cb_enpassant = game->ep;

	cptr = *cpptr;
	halfmove = game->halfmove;
	while (cptr = get_word(cptr, &len)) {

		/* parse the coordinate move */
		if (len != 4 && len != 5)
			break;
		if (!isfilelet(*cptr) || !isranknum(*(cptr+1)) ||
			!isfilelet(*(cptr+2)) || !isranknum(*(cptr+3)))
			break;

		from = to_offset(*cptr - 'a', *(cptr+1) - '1');
		to = to_offset(*(cptr+2) - 'a', *(cptr+3) - '1');
		cptr += 4;
		prom = 0;
		if (len == 5) {
			/* promotion */
			switch (*(u_char *)cptr++) {
			case 177:
			case 'K':
				prom = KING;
				break;
			case 178:
			case 'Q':
				prom = QUEEN;
				break;
			case 179:
			case 'N':
				prom = KNIGHT;
				break;
			case 180:
			case 'B':
				prom = BISHOP;
				break;
			case 181:
			case 'R':
				prom = ROOK;
				break;
			default:
				goto fail;
			}
		}

		/* generate the list of legal moves */
		gen_movelist(to_colour(halfmove));

		/* find the move in the list */
		for (i = 0; i < moveidx; i++) {
			moveptr = &movelist[i];
			if (moveptr->from == from && moveptr->to == to &&
			    moveptr->prom == moveptr->prom)
				break;
		}
		if (i == moveidx) {
			error("game %lu: illegal move \"%s\" in Fritz comment", game->num,
				format_move(from, to, prom));
			goto fail;
		}
        lastmove = movelist[i];
		lastmove_num = i;

		/* perform the move on the board */
		do_move(from, to, prom);
		move_txt = algebraic(to_colour(halfmove));

		if (halfmove == (int)game->halfmove)
			strcpy(pm_str, move_txt);
		strcat(pv_str, move_txt);
		strcat(pv_str, " ");
		halfmove++;
	}

	*cpptr = cptr;
	return 0;

fail:
	*cpptr = NULL;
	return -1;
}

/*
 * FRITZ_ACS
 *
 * Translate Fritz2 analysis time into seconds.
 */
static int
fritz_acs(cpptr)
	char **cpptr;
{
	int len, h, m, s;
	char *cptr;

	cptr = get_word(*cpptr, &len);
	if (!cptr)
		goto fail;
	if (sscanf(cptr, "(%02d:%02d:%02d)", &h, &m, &s) != 3)
		goto fail;

	*cpptr = cptr + len;
	sprintf(acs_str, "%lu",((u_long)h * 3600) + ((u_long)m * 60) + (u_long)s);
	return 0;

fail:
	*cpptr = NULL;
	return -1;
}

#define MOVES_SZ        1024		/* number of ChessBase moves allowed */
#define LINE_SZ			1024		/* size of EPD line */
#define NOPS			32			/* maximum number of supported EPD ops */
#define OP_NOTFOUND		0x1234		/* magic number used by dispatch_op() */
static u_char *moves, *mptr;		/* ChessBase moves */
static char *line;					/* EPD line */
static int year;					/* Year of game */
static int nmoves;					/* number of moves in game */
static int move_count;				/* used to calculate game header no. moves */
static int linenum;					/* line number in EPD file */
static int nops;					/* number of operands in EPD line */
static char *epd_op[NOPS];			/* pointers to EPD operand strings */

/*
 * IMPORT
 *
 * Convert an EPD file to ChessBase
 */
static int
import()
{
	int ret, epd_pos;
	u_long gamenum;
	time_t now;
	struct tm *tm;

	/* fix name of EPD file to avoid overflowing the source field */
	if (strlen(epdname) > 36) {
		epdname += strlen(epdname) - 36;
		strncpy(epdname, "...", 3);
	}

	/* Get year for game header */
	now = time(NULL);
	tm = localtime(&now);
	year = tm->tm_year;
	/* will this program still be running in the 21st century ??? */
	year += year < 94 ? 2000 : 1900;

	game = (Game)mem_alloc(sizeof(struct game));
	if (!game)
		return -1;
	moves = (u_char *)mem_alloc(MOVES_SZ);
	if (!moves) {
		free(game);
		return -1;
	}
	line = (char *)mem_alloc(LINE_SZ);
	if (!line) {
		free(moves);
		free(line);
		free(game);
		return -1;
	}
	gamenum = db->ngames + 1L;
	file_seek(db->cbf, read_index(db, gamenum));
	epd_pos = 1;
	linenum = 0;

    do {
		/* initialise variables */
		bzero((char *)game, sizeof(struct game));
		bzero((char *)moves, MOVES_SZ);
		nmoves = 0;
		move_count = 0;
		mptr = moves;

		/* set default players and source info */
		sprintf((char *)game->pinfo, "Position #%d", epd_pos++);

		/* read EPD line */
        ret = read_epd();
		if (ret)
			break;

		/* write ChessBase game */
        game->num = gamenum;
        game->mlen = nmoves;
		game->moves = moves;
		game->nmoves = (move_count + 1) / 2;
		game->clen = 0;
		sprintf((char *)game->sinfo, "EPD file \"%s\"", epdname);
		game->plen = ustrlen(game->pinfo);
		game->slen = ustrlen(game->sinfo);
        set_result(game->header, (4 << 2) | 3);
		game->year = year;
		if (write_game(db, game) < 0) {
			ret = -1;
			break;
		}
        db->ngames++;
        write_ngames(db);
        gamenum++;
		if (!quiet)
			output("line: %d\r", linenum);
    } while (ret == 0);

	free(line);
	free(moves);
    game_free(game);
    return ret;
}

/*
 * READ_EPD
 *
 * Read and interpret an EPD line.
 */
static int
read_epd()
{
	char *op;
	int ret;

	bzero(line, LINE_SZ);
	if (!fgets(line, LINE_SZ, file)) {
		if (ferror(file))
			error("error reading \"%s\"", epdname);
		return 1;
	}
	linenum++;

	op = parse_pos();
	if (!op) {
		error("%s(%d): error parsing position information", epdname,
		linenum);
		return -1;
	}
	if (!*op)
		return 0;			/* no operands; nothing to do */

	/* locate the operands and terminate at the semi-colon */
	for (nops = 0; nops < NOPS; nops++) {
		epd_op[nops] = get_word(op, NULL);
		if (!epd_op[nops] || *epd_op[nops] == '\n') {
			break;
		} else if (*epd_op[nops] == ';') {	/* empty */
			nops--;
			continue;
		}
		/* find the terminating semi-colon and replace it with a nul */
		while (*op && *op != ';' && *op != '\n')
			op++;
		*op++ = '\0';
	}
	epd_op[nops] = NULL;

	/* process the EPD operands */

	if (dispatch_op("fmvn", process_fmvn) < 0)
		return -1;

	if (dispatch_op("id", process_id) < 0)
		return -1;

	/* look for 'pv', 'pm' or 'bm' operands */
	ret = dispatch_op("pv", process_moveop);
	if (ret < 0)
		return -1;
	else if (ret == OP_NOTFOUND) {
		ret = dispatch_op("pm", process_moveop);
		if (ret < 0)
			return -1;
		else if (ret == OP_NOTFOUND) {
			ret = dispatch_op("bm", process_moveop);
			if (ret < 0)
				return -1;
		}
	}		

	if (ret == OP_NOTFOUND)
		ret = 0;			/* not an error */
	return ret;
}	

/*
 * PARSE_POS
 *
 * Parse the EPD position
 */
static char *
parse_pos()
{
	int i, brdfile, brdrank, colour, len;
	u_char piece, *board;
	char *args[4], *s, *ret;

	s = line;
	for (i = 0; i < 4; i++) {
		args[i] = get_word(s, &len);
		if (!args[i])
			return NULL;
		s = args[i] + len;
		*s++ = '\0';
	}
	ret = s;				/* return pointer to operands */

	board = game->board;
	bzero((char *) board, 64);

	s = args[0];
	for (brdrank = 7; brdrank >= 0; brdrank--, s++) {
		for (brdfile = 0; brdfile < 8 && *s && *s != '/'; brdfile++, s++) {
			if (isdigit(*s)) {
				brdfile += (*s - '0') - 1;
				continue;
			} else if (isalpha(*s)) {
				piece = text2piece(*s);
				if (piece != 0xff) {
					board[to_offset(brdfile, brdrank)] = piece;
					continue;
				}
			}
			return NULL;
		}
	}
	copy_board(board, cb_board);

	s = args[1];
	if (*s == 'w') {
		colour = 0;
	} else if (*s == 'b') {
		colour = 8;
	} else {
		return NULL;
	}

	s = args[2];
	if (*s != '-') {
		while (*s) {
			switch (*s) {
			case 'K':
				set_wcs(game->header, 1);
				break;
			case 'Q':
				set_wcl(game->header, 1);
				break;
			case 'k':
				set_bcs(game->header, 1);
				break;
			case 'q':
				set_bcl(game->header, 1);
				break;
			default:
				return NULL;
			}
			s++;
		}
	}

	s = args[3];
	if (*s != '-') {
		if (*s >= 'a' && *s <= 'h') {
			game->ep = *s - 'a' + 1;
			cb_enpassant = game->ep;
		} else {
			return NULL;
		}
		s++;
		if ((colour && *s != '3') || (!colour && *s != '6')) {
			return NULL;
		}
	}

	game->halfmove = to_halfmove(1, colour);
	set_partial(game->header);
	return ret;
}

/*
 * DISPATCH_OP
 *
 * Find EPD operands, and if found call the function to process the
 * operand.
 */
static int
dispatch_op(opname, func)
	char *opname;
	int (*func) P__((char *s));
{
	int i, len;

	len = strlen(opname);
	for (i = 0; i < nops; i++)
		if (strncmp(epd_op[i], opname, len) == 0)
			return (func)(epd_op[i]);
	return OP_NOTFOUND;
}

/*
 * PROCESS_MOVEOP
 *
 * Process 'pm', 'pv' and 'bm' operands.
 */
static int
process_moveop(opstr)
	char *opstr;
{
	int len, movenum, halfmove, one_only;
	MoveToken token;
	Move moveptr;
	char prev_char;

	if (strncmp(opstr, "bm", 2) == 0 || strncmp(opstr, "pm", 2) == 0)
		one_only = 1;		/* only process one move */
	else
		one_only = 0;

	opstr += 2;
	halfmove = game->halfmove;

	while (opstr = get_word(opstr, &len)) {
		prev_char = *(opstr+len);		/* save previous character */
		*(opstr+len) = '\0';			/* for error reporting */
		token = get_move_token(opstr);
		if (token == MT_ERROR) {
			error("%s(%d): unrecognised move \"%s\"", epdname, linenum,
			    opstr);
			return -1;
		}
		movenum = parse_move(token, opstr, halfmove, epdname, linenum);
		if (movenum < 0)
			return -1;
		moveptr = &movelist[movenum];
		do_move(moveptr->from, moveptr->to, moveptr->prom);
		if (add_move((u_char) (movenum + 1)) < 0)
				return -1;
		move_count++;
		*(opstr+len) = prev_char;			/* restore previous character */
		opstr += len;
		halfmove++;

		if (one_only)
			return 0;
	}
	return 0;
}

/*
 * PROCESS_FMVN
 *
 * Process 'fmvn' operands.
 */
static int
process_fmvn(opstr)
	char *opstr;
{
	int move, colour;

	colour = to_colour(game->halfmove);	/* already set by parse_pos() */
	opstr = get_word(opstr + 4, NULL);
	if (!opstr)
		goto fail;
	move = (int)xatol(opstr);
	if (xatol_ptr && *xatol_ptr != ' ')
		goto fail;
	game->halfmove = to_halfmove(move, colour);
	return 0;
fail:
	error("%s(%d): fmvn operand error", epdname, linenum);
	return -1;
}

/*
 * PROCESS_ID
 *
 * Process 'id' operand.
 */
static int
process_id(opstr)
	char *opstr;
{
	int len;

	opstr = get_word(opstr + 2, &len);
	if (!opstr)
		goto fail;

	if (len < 2 || *opstr != '"' || *(opstr+len-1) != '"')
		goto fail;

	*(opstr+len) = '\0';
	sprintf((char *)game->pinfo, "ID: %s", opstr);
	return 0;
fail:
	error("%s(%d): id operand error", epdname, linenum);
	return -1;
}

/*
 * ADD_MOVE
 *
 * Add a move to the ChessBase movelist.
 */
static int
add_move(move)
    u_char move;
{
    if (nmoves >= MOVES_SZ) {
        error("ChessBase move buffer full!");
        return -1;
    }
    *mptr++ = move;
    nmoves++;
    return 0;
}
