/***************************************************************************
 * CBCHECK -- Check games in a ChessBase file.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          11/11/93        Initial.
 * 1.1          20/03/94        Added duplicate checking.
 * 1.1a			21/11/94		Fixed bug when reading names of databases
 *								from a listfile.
 ***************************************************************************/

#include "global.h"

#define BANNER "CBCHECK  Copyright (c)1993-94 Andy Duplain  "
#ifdef ANSI_C
#define VERSION() output("V1.1a [%s %s]\n", __DATE__, __TIME__);
#else
#define VERSION() output("V1.1a\n");
#endif

static void usage P__((void));

static int quiet = 0;                  /* -q flag */
static int dupcheck = 0;               /* -d flag */
static u_long first = 1L, last = 0xffffffffL;
static char *dupop = NULL;
static char *listfile = NULL;

static int check P__((char *dbname));
static int do_dupcheck P__((FILE * ofp, Game game1, Game game2));

static void
usage()
{
    error("usage: cbcheck [options] dbspec [... dbspec]");
    error("options:");
    error("  -d outfile\tcheck for duplicate games");
    error("  -f listfile\tread from list of additional databases to check");
    error("  -q\t\tquiet");
    error("  -r x-y\tspecify the first and last game to check");
    exit(1);
}

int
main(argc, argv)
    int argc;
    char **argv;
{
    int c, status, totalerr, nfiles;
    char *name;
    FILE *lfp;

    opterr = 0;
    while ((c = getopt(argc, argv, "d:f:qr:")) != EOF) {
        switch (c) {
        case 'd':
            dupcheck++;
            dupop = optarg;
            break;
        case 'f':
            listfile = optarg;
            break;
        case 'q':
            quiet++;
            break;
        case 'r':
            first = range_first(optarg);
            last = range_last(optarg);
            if ((!first || !last) || first > last) {
                error("invalid range");
                return 1;
            }
            break;
        case '?':
        default:
            usage();
        }
    }
    argc -= optind;
    argv += optind;

    if (!listfile && !argc)
        usage();

    if (!quiet) {
        output(BANNER);
        VERSION();
    }
    totalerr = 0;
    nfiles = 0;
    while (argc) {
        if (find_database_init(*argv) < 0)
            return 1;
        while (name = find_database()) {
            status = check(name);
            if (status >= 0) {
                totalerr += status;
                nfiles++;
            }
        }
        find_database_exit();
        argc--;
        argv++;
    }

    if (listfile) {
        lfp = fopen(listfile, "rt");
        if (!lfp) {
            error("error opening listfile \"%s\"", listfile);
        } else {
            while (name = read_listfile(lfp)) {
                status = check(name);
                if (status >= 0) {
                    totalerr += status;
                    nfiles++;
                }
            }
            fclose(lfp);
        }
    }
    if (!quiet && nfiles)
        output("files checked: %d, total errors: %d\n", nfiles,
          totalerr);
    return totalerr ? 2 : 0;
}

/*
 * CHECK
 *
 * Check the actual database
 *
 * Returns -1 if error opening database or allocating memory,
 *         0 if no errors found.
 *         else the number of errors found is returned.
 */
static int
check(dbname)
    char *dbname;
{
    Database db;
    Game game, game2;
    u_long i, j, ngames, dbfirst, dblast;
    int nerr, progress, last_report, eco_missing;
    int do_precheck, eco_code, eco_code2;
    FILE *dupfp;
    u_char *hdr;

    eco_missing = 0;
    db = open_database(dbname);
    if (!db)
        return -1;

    if (dupcheck) {
        dupfp = fopen(dupop, "at");
        if (!dupfp) {
            error("error opening \"%s\"", dupop);
            close_database(db);
            return -1;
        }
        fprintf(dupfp, "%s\n", dbname);
    }
    if (last > db->ngames)
        dblast = db->ngames;
    else
        dblast = last;
    if (first > dblast)
        dbfirst = dblast;
    else
        dbfirst = first;

    game = (Game) mem_alloc(sizeof(struct game));
    if (!game) {
        close_database(db);
        return -1;
    }
    if (dupcheck) {
        game2 = (Game) mem_alloc(sizeof(struct game));
        if (!game2) {
            game_free(game);
            close_database(db);
            return -1;
        }
    }
    nerr = 0;
    last_report = -1;
    ngames = (dblast - dbfirst) + 1;

    if (!quiet)
        output("checking %s\n", dbname);

    for (i = dbfirst; i <= dblast; i++) {
        game->num = i;
        if (read_game(db, game) < 0) {
            nerr++;
            continue;
        }
        if (is_deleted(game->header))
            continue;
        if (process_moves(game, NULL) < 0)
            nerr++;
        if (dupcheck && is_full(game->header)) {
            no_error = 1;              /* suppress error messages here */
            if (game->eco_letter) {
                do_precheck = 1;
                eco_code = get_eco1(game->header);
            } else {
                do_precheck = 0;
                eco_code = 0;
            }

            for (j = i + 1L; j <= dblast; j++) {
                game2->num = j;

            /* save time by checking the year and ECO code from the game
               header */
                if (do_precheck) {
                    if (read_header(db, game2) < 0)
                        continue;
                    hdr = game2->header;

                    if (!is_full(hdr) || is_deleted(hdr))
                        continue;

                    eco_code2 = get_eco1(hdr);
                    if (eco_code2 &&
                      eco_code2 != eco_code)
                        continue;

                    if (hdr[0] != 127 &&
                      game->year != (u_short) (1900 + (char) hdr[0]))
                        continue;
                }
                if (read_game(db, game2) < 0)
                    continue;
                do_dupcheck(dupfp, game, game2);
                game_tidy(game2);
            }
            no_error = 0;
        }
    /* if the game is "full", check it has an ECO code */
        if (is_full(game->header) && !game->eco_letter)
            eco_missing = 1;

    /* report progess of operation */
        if (!quiet) {
            progress = (int) (((i - dbfirst + 1L) * 100L) /
              ((dblast - dbfirst) + 1));
            if (last_report != progress) {
                output("%d%%\r", progress);
                last_report = progress;
            }
        }
        game_tidy(game);
    }

    if (dupcheck)
        game_free(game2);
    game_free(game);
    if (dupcheck)
        fclose(dupfp);
    close_database(db);
    if (!quiet)
        output("errors: %d%s\n", nerr,
          eco_missing ? " (some games have no ECO codes)" : "");
    return nerr;
}

/*
 * DO_DUPCHECK
 *
 * Compare the main-line moves from two games, to see if they're the same
 * game.  If the game moves appear to match, attempt to match the year,
 * players and source information to discover if the games are in fact
 * duplicate.
 */
static int
do_dupcheck(ofp, game1, game2)
    FILE *ofp;
    Game game1, game2;
{
    int len1, len2;
    u_char *mptr1, *mptr2;

    len1 = (int) game1->mlen;
    if (!len1)
        return 0;
    mptr1 = game1->moves;
    len2 = (int) game2->mlen;
    if (!len2)
        return 0;
    mptr2 = game2->moves;

    if (match_moves(mptr1, len1, mptr2, len2) == 0)
        return 0;

 /* output the match string */

    fprintf(ofp, "%lu / %lu ", game1->num, game2->num);

    if (get_nmoves(game1->header) == get_nmoves(game2->header)) {
        fprintf(ofp, "m= ");

    /* see if any variations exist, and how many match */
        mptr1 = (u_char *) memchr(game1->moves, 0xff, game1->mlen);
        mptr2 = (u_char *) memchr(game2->moves, 0xff, game2->mlen);
        if (mptr1 && mptr2) {
            if (game1->mlen == game2->mlen)
                fprintf(ofp, "v= ");
            else if (game1->mlen < game2->mlen)
                fprintf(ofp, "v< ");
            else
                fprintf(ofp, "v> ");
        } else if (mptr1 || mptr2) {
            fprintf(ofp, "v~ ");
        }
    } else if (get_nmoves(game1->header) < get_nmoves(game2->header))
        fprintf(ofp, "m< ");
    else
        fprintf(ofp, "m> ");

    fputc('\n', ofp);
    fflush(ofp);
    return 1;
}
