/***************************************************************************
 * CBASCII -- Convert ChessBase games to and from ASCII files.
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.6			17/10/95		Added support for short-format pawn captures
 *								(<file><file>) and promotions (<file><file><piece>).
 * 1.5          31/07/95        Added -l and -F option.
 * 1.4          29/11/94        Added -L option and diagram processing.
 * 1.0          11/11/93        Initial.
 ***************************************************************************/

#include "global.h"
#include "aexport.h"
#include "aimport.h"
#include "cbascii.h"
#include <ctype.h>

#ifdef WIN32
#define BANNER "CBASCII/32  Copyright (c)1993-95 Andy Duplain (duplaina@syntegra.bt.co.uk)\n"
#else
#define BANNER "CBASCII  Copyright (c)1993-95 Andy Duplain (duplaina@syntegra.bt.co.uk)\n"
#endif

#ifdef WIN32
#define VERSION() output("Public Domain.  V1.6 (Win32) [%s %s]\n", __DATE__, __TIME__);
#else /* WIN32 */
#ifdef ANSI_C
#define VERSION() output("Public Domain.  V1.6 [%s %s]\n", __DATE__, __TIME__);
#else /* ANSI_C */
#define VERSION() output("Public Domain.  V1.6\n");
#endif /* ANSI_C */
#endif /* WIN32 */

/* "conv" defines */
#define EXPORT          1
#define IMPORT          2

int conv = 0;                          /* conversion */
int comm_strip = 0;                    /* strip comments */
int var_strip = 0;                     /* strip variations */
int format = 0;                        /* import / export format */
int diagtype = DIAG_8X8;               /* -d flag */
char nat_black_sq = '+';               /* "Natural" diagram black-square type */
int append = 0;                        /* -a flag */
int quiet = 0;                         /* -q flag */
int testmode = 0;                      /* -t flag */
int ignore_errors = 0;                 /* -I flag */
int dont_enforce = 0;                  /* -E flag */
u_long skiplines = 0;                  /* -L flag */
int long_comments = 0;                 /* -l flag */
int no_nag = 0;                        /* PGN without NAGs */
u_long first = 1, last = 0xffffffffL;
char *dbname;
char *asciiname;
Database db = NULL;
FILE *file;
char comm_start = 0;                   /* comment start character */
char comm_end = 0;                     /* comment end character */
char var_start = 0;                    /* variation start character */
char var_end = 0;                      /* variation end character */
char *ourpath;                         /* path to this utility */

/* diagram characters */
u_char diag_top_left_corner = '+';
u_char diag_top_right_corner = '+';
u_char diag_bottom_left_corner = '+';
u_char diag_bottom_right_corner = '+';
u_char diag_top_border = '-';
u_char diag_left_border = '|';
u_char diag_right_border = '|';
u_char diag_bottom_border = '-';
u_char diag_white_square = '-';
u_char diag_black_square = '+';
u_char diag_white_pieces[16] = "*KQNBRP**KQNBRP*";
u_char diag_black_pieces[16] = "*kqnbrp**kqnbrp*";
u_char diag_space_between_squares = 0;
u_char diag_borders = 1;

/* diagram keywords and pointers */
struct {
	char *keyword;
	u_char *data;
} defs[] = {
	{ "top_left_corner", &diag_top_left_corner },
	{ "top_right_corner", &diag_top_right_corner },
	{ "bottom_left_corner", &diag_bottom_left_corner },
	{ "bottom_right_corner", &diag_bottom_right_corner },
	{ "top_border", &diag_top_border },
	{ "left_border", &diag_left_border },
	{ "right_border", &diag_right_border },
	{ "bottom_border", &diag_bottom_border },
	{ "white_square", &diag_white_square },
	{ "black_square", &diag_black_square },
	{ "white_king_white_square", &diag_white_pieces[1] },
	{ "white_queen_white_square", &diag_white_pieces[2] },
	{ "white_knight_white_square", &diag_white_pieces[3] },
	{ "white_bishop_white_square", &diag_white_pieces[4] },
	{ "white_rook_white_square", &diag_white_pieces[5] },
	{ "white_pawn_white_square", &diag_white_pieces[6] },
	{ "white_king_black_square", &diag_white_pieces[9] },
	{ "white_queen_black_square", &diag_white_pieces[10] },
	{ "white_knight_black_square", &diag_white_pieces[11] },
	{ "white_bishop_black_square", &diag_white_pieces[12] },
	{ "white_rook_black_square", &diag_white_pieces[13] },
	{ "white_pawn_black_square", &diag_white_pieces[14] },
	{ "black_king_white_square", &diag_black_pieces[1] },
	{ "black_queen_white_square", &diag_black_pieces[2] },
	{ "black_knight_white_square", &diag_black_pieces[3] },
	{ "black_bishop_white_square", &diag_black_pieces[4] },
	{ "black_rook_white_square", &diag_black_pieces[5] },
	{ "black_pawn_white_square", &diag_black_pieces[6] },
	{ "black_king_black_square", &diag_black_pieces[9] },
	{ "black_queen_black_square", &diag_black_pieces[10] },
	{ "black_knight_black_square", &diag_black_pieces[11] },
	{ "black_bishop_black_square", &diag_black_pieces[12] },
	{ "black_rook_black_square", &diag_black_pieces[13] },
	{ "black_pawn_black_square", &diag_black_pieces[14] },
	{ "space_between_squares", &diag_space_between_squares },
	{ "borders", &diag_borders }
};

#define NDEFS (sizeof(defs) / sizeof(defs[0]))

static void usage P__((void));
static int parse_char_arg P__((char *s, char *type));
static void set_char P__((char *var, char value));
static int parse_format_arg P__((char *s));
static int parse_diag_arg P__((char *s));

static void
usage()
{
#ifdef WIN32
	error("usage: cbasc32 -e|-i [options] ascii-file database");
#else
	error("usage: cbascii -e|-i [options] ascii-file database");
#endif
	error("  -e = export (ChessBase to ASCII)   -i = import (ASCII to ChessBase)");
	error("options:");
	error("  -F file\tOutput error messages to file, rather than screen");
	error("  -r x-y\tspecify first and last game to convert");
	error("  -q\t\tquiet");
	error("  -s\t\tstrip comments");
	error("  -S\t\tstrip variations");
	error("export-only options:");
	error("  -a\t\topen ascii-file in \"append\" mode");
	error("  -c char|value\tspecify 'comment start' character");
	error("  -C char|value\tspecify 'comment end' character");
	error("  -d type\tspecify diagram type (default is \"8x8\")");
	error("  -f format\tspecify export format (default is \"pgn\")");
	error("  -k\t\tIgnore ChessBase game checksum errors");
	error("  -v char|value\tspecify 'variation start' character");
	error("  -V char|value\tspecify 'variation end' character");
	error("import-only options:");
	error("  -E\t\tdon't enforce 180-move limit (for CB-for-Windows/Atari-CB)");
	error("  -I\t\tmake errors non-fatal (ignore errors)");
	error("  -l\t\tallow long comments (for CB-for-Windows)");
	error("  -L line\tstart reading ascii-file at line given (1 onwards)");
	error("  -t\t\ttest mode; parse ascii-file without adding to database");
	exit(1);
}

/*
 * PARSE_CHAR_ARG
 *
 * Get the comment/variation character specifier as either a character or
 * as a decimal value
 */
static int
parse_char_arg(s, type)
	char *s, *type;
{
	int ret;

	if (strlen(s) == 3 && *s == '\'' && *(s + 2) == '\'')
		return *(s + 1);               /* character */

	ret = (int) xatol(s);
	if (!xatol_err)
		return ret & 0xff;

	error("invalid %s character specifier \"%s\"", type, s);
	error("characters should be specified as either a character enclosed");
	error("in quotes (e.g 'A') or as its decimal value (e.g 65)");
	exit(1);
}

/*
 * SET_UCHAR
 *
 * Set the value of an unsigned character, but only if it is currently
 * unset (used to set comment and variation characters).
 */
static void
set_char(var, value)
	char *var;
	char value;
{
	if (!*var)
		*var = value;
}

/*
 * PARSE_FORMAT_ARG
 *
 * Parse format-type argument
 */
static int
parse_format_arg(s)
	char *s;
{
	if (strcmp(s, "cb") == 0) {
		format = TYPE_CB;
		set_char(&comm_start, 0);
		set_char(&comm_end, 0);
		set_char(&var_start, '[');
		set_char(&var_end, ']');
		return 0;
	}
	if (strcmp(s, "pgn") == 0) {
		format = TYPE_PGN;
		set_char(&comm_start, '{');
		set_char(&comm_end, '}');
		set_char(&var_start, '(');
		set_char(&var_end, ')');
		no_nag = 0;
		return 0;
	}
	if (strcmp(s, "pgn_no_nag") == 0) {
		format = TYPE_PGN;
		set_char(&comm_start, '{');
		set_char(&comm_end, '}');
		set_char(&var_start, '(');
		set_char(&var_end, ')');
		no_nag = 1;
		return 0;
	}
	error("invalid format specifier: valid specifiers are:");
	error("\"cb\"\t\tChessBase-format");
	error("\"pgn\"\t\tPortable Game Notation");
	error("\"pgn_no_nag\"\tPortable Game Notation (without NAGs)");
	exit(1);
}

/*
 * PARSE_DIAG_ARG
 *
 * Parse diagram-type argument
 */
static int
parse_diag_arg(s)
	char *s;
{
	FILE *def_file;
	int i, len, linenum;
	static char buffer[128];
	char *cptr;
	u_long val;

	if (strcmp(s, "cb") == 0) {
		diagtype = DIAG_CB;
		return 0;
	}
	if (strcmp(s, "8x8") == 0) {
		diagtype = DIAG_8X8;
		return 0;
	}
	if (strncmp(s, "8x8=", 4) == 0) {
		diagtype = DIAG_8X8;
		s += 4;
		def_file = fopen(s, "rt");
		/* cannot find the file; look in the same directory as this
		   utility */
		if (!def_file && ourpath) {
			char *newpath;
			newpath = mem_alloc(strlen(ourpath) + strlen(s) +
				sizeof(PATHSEP) + 1);
			if (!newpath)
				exit(1);
			strcpy(newpath, ourpath);
			newpath[strlen(ourpath)] = PATHSEP;
			strcat(newpath, s);
			def_file = fopen(newpath, "rt");
			free(newpath);
		}
		if (!def_file) {
			error("couldn't open definition file \"%s\"", s);
			exit(1);
		}

		linenum = 0;
		while(fgets(buffer, sizeof(buffer), def_file)) {
			linenum++;
			char_change(buffer, '\n', '\0');
			if (buffer[0] == '#' || buffer[0] == '\0')
				continue;
			cptr = get_word(buffer, &len);
			if (!len)
				continue;
			for (i = 0; i < NDEFS; i++) {
				if (strncmp(cptr, defs[i].keyword, len) == 0)
					break;
			}             
			if (i == NDEFS) {
				*(cptr + len) = '\0';
				error("%s(%d): invalid keyword \"%s\"", s, linenum, cptr);
				fclose(def_file);
				exit(1);
			}

			cptr = get_word(cptr + len, &len);
			if (!len) {
				error("%s(%d): no value found!", s, linenum);
				fclose(def_file);
				exit(1);
			}

			if (len == 3 && *cptr == '\'' && *(cptr+2) == '\'') { 
				*(defs[i].data) = *(cptr+1);
			} else {    /* decimal value */
				val = xatol(cptr);
				if (xatol_err  && !isspace(*xatol_ptr)) {
					*(cptr + len) = '\0';
					error("%s(%d): invalid value \"%s\"",
						s, linenum, cptr);
					goto failed;
				}
				if (val & ~0xff) {
					error("%s(%d): decimal value out-of-range (%ld)",
						s, linenum, val);
					goto failed;
				}
				*(defs[i].data) = (u_char)val;
			}
		}
		fclose(def_file);
		return 0;
	}
	error("invalid diagram-type specifier: valid specifiers are:");
	error("\"cb\"\t\tChessBase-format");
	error("\"8x8\"\t\t8x8 diagram");
	error("\"8x8=file\"\t8x8 diagram with definitions from file");
	exit(1);

failed:
	fclose(def_file);
	exit(1);
}

int
main(argc, argv)
	int argc;
	char **argv;
{
	int c, ret;
	char *mode, *cptr;

	ret = 0;

	if (cptr = rindex(*argv, PATHSEP)) {
		*cptr = '\0';
		ourpath = xstrdup(*argv);
		if (!ourpath)
			exit(1);
		*cptr = PATHSEP;
	}

	opterr = 0;
	while ((c = getopt(argc, argv, "ac:C:d:ef:iEF:kL:lIqr:sStv:V:")) != EOF) {
		switch (c) {
		case 'a':
			append++;
			break;
		case 'c':
			comm_start = parse_char_arg(optarg, "comment");
			break;
		case 'C':
			comm_end = parse_char_arg(optarg, "comment");
			break;
		case 'd':
			parse_diag_arg(optarg);
			break;
		case 'e':
			if (conv) {
				error("specify either \"-e\" or \"-i\"");
				return 1;
			}
			conv = EXPORT;
			break;
		case 'f':
			parse_format_arg(optarg);
			break;
		case 'i':
			if (conv) {
				error("specify either \"-e\" or \"-i\"");
				return 1;
			}
			conv = IMPORT;
			break;
		case 'E':
			dont_enforce++;
			break;
		case 'F':
			fp_error = fopen(optarg, "at");
			if (!fp_error) {
				fp_error = stdout;
				error("Couldn't open error file \"%s\"; using standard output",
					optarg);
			}
			break;
		case 'I':
			ignore_errors++;
			break;
		case 'L':
			skiplines = (u_long)strtol(optarg, &cptr, 10);
			if (*cptr) {
				error("invalid line number \"%s\"", optarg);
				return 1;
			}
			if (skiplines == 0) {
				error("line number must start at 1");
				return 1;
			}
			break;
		case 'k':
			ignore_chksum_err++;
			break;
		case 'l':
			long_comments++;
			break;
		case 'q':
			quiet++;
			break;
		case 'r':
			first = range_first(optarg);
			last = range_last(optarg);
			if ((!first || !last) || first > last) {
				error("invalid range");
				return 1;
			}
			break;
		case 's':
			comm_strip++;
			break;
		case 'S':
			var_strip++;
			break;
		case 't':
			testmode++;
			break;
		case 'v':
			var_start = parse_char_arg(optarg, "variation");
			break;
		case 'V':
			var_end = parse_char_arg(optarg, "variation");
			break;
		case '?':
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (!conv) {
		error("you must specify \"-e\" to export or \"-i\" to import");
		return 1;
	}
	if (argc != 2)
		usage();

	if (!format) {
		format = TYPE_PGN;
		set_char(&comm_start, '{');
		set_char(&comm_end, '}');
		set_char(&var_start, '(');
		set_char(&var_end, ')');
	}
	asciiname = *argv++;
	dbname = *argv;
	kill_ext(dbname);

	if (!quiet) {
		output(BANNER);
		VERSION();
	}
	no_error = 1;
	db = open_database(dbname);
	no_error = 0;
	if (!db) {
		if (conv == IMPORT) {
			if (!quiet)
				output("creating database %s\n", dbname);
			db = create_database(dbname);
			if (!db)
				return 1;
		} else {                       /* exporting */
			error("error opening database %s", dbname);
			return 1;
		}
	}
	if (conv == EXPORT) {
		if (last > db->ngames)
			last = db->ngames;
		if (first > last)
			first = last;
	}
 /* open the ASCII file */
	if (conv == IMPORT)
		mode = "rt";
	else
		mode = append ? "at" : "wt";
	file = fopen(asciiname, mode);
	if (!file) {
		error("error opening \"%s\"", asciiname);
		return 1;
	}
	if (conv == EXPORT) {
		ret = export();
	} else {                           /* importing */
		ret = import();
	}

	close_database(db);
	fclose(file);
	if (fp_error && fp_error != stdout)
		fclose(fp_error);

	return ret;
}
