/***************************************************************************
 * AIMPORT.C -- Import ASCII data to ChessBase
 *
 * Copyright (c)1993 Andy Duplain.
 *
 * Version      Date            Comments
 * =======      ====            ========
 * 1.0          11/01/94        Initial.
 * 1.1          22/01/94        Added check for ambiguous input move, and
 *                              changed name of variables so as not to
 *                              clash with to_file() and to_rank()!!!
 *                              Added tidy_name().
 * 1.2          09/04/94        Added call to char_change() from tidy_name().
 * 1.3          30/04/94        Added call to tidy_string() from
 *                              get_pgn_header() (for Event/Site), and from
 *                              tidy_name().
 * 1.4          29/11/94        Added -L option.  Added "skipped" output
 *                              when skipping games outside range.
 * 1.5          31/07/95        New -l option for CBWIN. PGN comments can
 *                              be free-format and long.
 *				22/08/95		Made all variables relating to PGN line
 *								number u_long.  Increased size of move and
 *								comment buffers.
 ***************************************************************************/

#include "global.h"
#include "aimport.h"
#include "cbascii.h"
#include "ascan.h"
#include "nag.h"
#include <ctype.h>

/* add_to_field() "type" defines */
#define SOURCE_FIELD    1              /* add to source field */
#define PLAYER_FIELD    2              /* add to player field */

/* "pos" defines */
#define MOVES_SZ        0x1000
#define COMMS_SZ        0x7fff
static Game game;
static u_char *moves, *moves_ptr;
static unsigned nmoves;
static u_char *comms, *comms_ptr;
static unsigned ncomms;
static char *init_comm;
static u_char *last_comm;
static int comm_full;
static char *wplayer, *bplayer, *event, *site, *round, *annotator;
static int varlev;
static unsigned next_report;
static unsigned game_count;
static unsigned move_count;
static u_long game_start;         /* line when game started */
static int in_game;
static int init_halfmove;
#define REPORT_INC 10;
static u_char eval, poseval, moveval;
int empty_comms_skipped = 0;      /* number of empty comments skipped */

static void setup_game P__((void));
static int parse_game P__((unsigned halfmove));
static int add_move P__((u_char move));
static int get_file_rank P__((char *s));
static int get_pgn_header P__((int token, char *s));
static void tidy_name P__((char *s));
static int add_info_data P__((void));
static int add_to_field P__((int field, char *data, char *sep, char leadchar,
	char endchar));
static int parse_fen P__((char *s));
static void decode_nag P__((char *s));
static int flush_eval P__((void));
static int add_comment P__((char *s));
static int add_comm P__((u_char comm));

/*
 * IMPORT
 */
int
import()
{
	int res, nerrors;
	u_long gamenum;

	game = (Game) mem_alloc(sizeof(struct game));
	if (!game)
		return -1;
	moves = (u_char *) mem_alloc(MOVES_SZ);
	if (!moves) {
		free(game);
		return -1;
	}
	comms = (u_char *) mem_alloc(COMMS_SZ);
	if (!comms) {
		free(game);
		free(moves);
		return -1;
	}
	gamenum = db->ngames + 1L;
	file_seek(db->cbf, read_index(db, gamenum));

	linenum = 1;
	if (skiplines) {
		while (linenum < skiplines) {
			int c;
			do {
				c = getc(file);
			} while (c != '\n' && c != EOF);
			if (c == EOF) {
				error("lineskip value beyond end-of-file");
				return 1;
			}
			linenum++;
		}
		if (!quiet)
			output("starting on line %lu\n", linenum);
	}

	yyin = file;
	next_report = REPORT_INC;
	game_count = 0;
	nerrors = 0;

	do {
		setup_game();
		res = parse_game(1);
		if (res == 0 || (ignore_errors && res == -1))
			game_count++;

		if (res == 0 && !testmode &&
		  ((unsigned)first <= game_count && (unsigned)last >= game_count)) {

			game->num = gamenum;
			game->mlen = nmoves;
			if (ncomms > 1)
				game->clen = ncomms;
			if (write_game(db, game) < 0) {
				res = -1;
			}
			db->ngames++;
			write_ngames(db);
			gamenum++;
		}
		if (ignore_errors && res == -1) {
			nerrors++;
			res = 0;
		}
	} while (res == 0);

	if (!quiet) {
		if ((unsigned)first <= game_count && (unsigned)last >= game_count)
			output("game: %u   \n", game_count);
		else
			output("skipped: %u\n", game_count);
		if (ignore_errors) {
			output("%d error%s\n", nerrors,
			  nerrors == 1 ? "" : "s");
		}
        if (empty_comms_skipped) {
            output("%d \"empty\" comments discarded\n", empty_comms_skipped);
        }
	}
	if (init_comm)
		free(init_comm);
	game_free(game);
	return 0;
}

/*
 * SETUP_GAME
 *
 * Setup the game structure and reset global variables.
 */
static void
setup_game()
{
	bzero((char *) game, sizeof(struct game));
	bzero((char *) moves, MOVES_SZ);
	bzero((char *) comms, COMMS_SZ);
	game->moves = moves;
	game->comments = comms;
	game->halfmove = 1;
	nmoves = 0;
	ncomms = 1;
	moves_ptr = moves;
	comms_ptr = comms;
	*comms_ptr++ = 0xff;               /* set initial marker */
	if (init_comm)
		free(init_comm);
	init_comm = NULL;
	last_comm = NULL;
	comm_full = 0;
	init_board(cb_board);
	cb_enpassant = 0;
	varlev = 0;
	in_game = 0;
	init_halfmove = 0;
	move_count = 0;
	wplayer = NULL;
	bplayer = NULL;
	event = NULL;
	site = NULL;
	round = NULL;
	annotator = NULL;
	eval = 0;
	poseval = 0;
	moveval = 0;
	game_start = 0;
}

/*
 * PARSE_GAME
 *
 * Parse the moves in a game.
 *
 * Returns: 0 if end of game or variation found, 1 if end of file found,
 *          -1 if error.
 */
static int
parse_game(halfmove)
	unsigned halfmove;
{
	u_char saved_board[64], prev_board[64];
	int saved_ep, prev_ep;
	int i, c, colour, ignoring;

	if (varlev && !var_strip)
		if (add_move(0xff) < 0)
			return -1;

	ignoring = 0;
	while (c = yylex()) {

		/* ignore everything until a game header is seen */
		if (!in_game && !(c >= A_PGN_HEADER_S && c <= A_PGN_HEADER_E))
			continue;

		/* if ignoring because more than 180 moves have been seen, then
		   just look for end-of-variation or end-of-game markers */
		if (ignoring) {
			if (c == A_VAREND || (c >= A_RESULT_S && c <= A_RESULT_E))
				break;
			else
				continue;
		}

		if (!quiet && game_count >= next_report) {
			if ((unsigned)first <= game_count && (unsigned)last >= game_count)
				output("game: %u   \r", game_count);
			else
				output("skipped: %u\r", game_count);
			next_report += REPORT_INC;
		}
		colour = to_colour(halfmove);

		if (c >= A_PGN_HEADER_S && c <= A_PGN_HEADER_E) {
			if (varlev) {
				error("%s(%lu): header in middle of game!",
				  asciiname, linenum);
				return -1;
			}
			if (get_pgn_header(c, yytext) < 0)
				return -1;
			if (!game_start)
				game_start = linenum;
			if (init_halfmove)
				halfmove = init_halfmove;
			in_game = 1;
		} else if (c >= A_MOVENUM_S && c <= A_MOVENUM_E) {
			i = atoi(yytext);
			if (!i) {
				error("%s(%lu): invalid move number \"%s\"",
				  asciiname, linenum, yytext);
				if (game_start)
					error("game started on line %lu", game_start);
				return -1;
			}
			i = to_halfmove(i, c == A_BLACK_MOVENUM);
			if ((int)halfmove != i) {
				error("%s(%lu): expected \"%s\" rather than \"%s\"",
				  asciiname, linenum, format_movenum(halfmove), yytext);
				if (game_start)
					error("game started on line %lu", game_start);
				return -1;
			}
			/* enforce 180-move limit...
			   (main-line max: 180 black, variation max: 180 white) */
			if (!dont_enforce &&
			  ((int)(halfmove - init_halfmove) > to_halfmove(180, !varlev))) {
				if (!quiet)
					error("%s(%lu): warning: %s exceeds 180 moves; truncating",
						asciiname, linenum,
						varlev ? "variation" : "main-line");
				ignoring = 1;
				continue;
			}
		} else if (c >= A_MOVE_S && c <= A_MOVE_E) {
			int movenum;
			Move moveptr;

			c -= A_MOVE_S;      /* convert lex token into a MoveToken */
			flush_eval();

			copy_board(cb_board, prev_board);
			prev_ep = cb_enpassant;

			if (varlev && var_strip) {
				halfmove++;
				continue;
			}
			movenum = parse_move((MoveToken)c, yytext, halfmove, asciiname, linenum);
			if (movenum < 0)
				return -1;
			moveptr = &movelist[movenum];
			do_move(moveptr->from, moveptr->to, moveptr->prom);
			if (add_move((u_char) (movenum + 1)) < 0)
				return -1;

			if (!varlev)
				move_count++;
			halfmove++;

		} else if (c >= A_RESULT_S && c <= A_RESULT_E) {
			break;
		} else if (c == A_COMMENT) {
			if (comm_strip || (varlev && var_strip)) {
				eval = 0;
				poseval = 0;
				moveval = 0;
				continue;
			}
			if (!nmoves) {
				if (!init_comm) {
					init_comm = xstrdup(yytext);
					if (!init_comm)
						return -1;
				}
			} else {
				if (add_comment(yytext) == 0)
					*(moves_ptr - 1) |= 0x80;
			}
		} else if (c >= A_EVAL_S && c <= A_EVAL_E) {
			switch (c) {
			case A_GOOD_MOVE:
				eval = 1;
				break;
			case A_BAD_MOVE:
				eval = 2;
				break;
			case A_INTERESTING_MOVE:
				eval = 3;
				break;
			case A_DUBIOUS_MOVE:
				eval = 4;
				break;
			case A_BRILLIANT_MOVE:
				eval = 5;
				break;
			case A_BLUNDER_MOVE:
				eval = 6;
				break;
			case A_NAG:
				decode_nag(yytext);
				break;
			case A_NAG_MATE:
			case A_NAG_NOVELTY:
				/* ChessBase-for-Windows 1.1. */
				/* These NAGs are illegal, so we will just ignore them */
				break;
			default:
				break;
			}
			continue;
		} else if (c == A_VARSTART) {
			if (!nmoves) {
				error("%s(%lu): no move for variation!",
				  asciiname, linenum);
				if (game_start)
					error("game started on line %lu", game_start);
				return -1;
			}
			flush_eval();

		/* put the position before the last move was parsed back into the
		   global board */
			copy_board(cb_board, saved_board);
			saved_ep = cb_enpassant;
			copy_board(prev_board, cb_board);
			cb_enpassant = prev_ep;

			varlev++;
			if (parse_game(halfmove - 1) < 0)
				return -1;
			varlev--;

		/* restore position before variation */
			copy_board(saved_board, cb_board);
			cb_enpassant = saved_ep;
		} else if (c == A_VAREND) {
			break;
		} else {
			if (c == '{') {
				error("%s(%lu): broken comment (unmatched braces ?)",
				  asciiname, linenum);
				if (game_start)
					error("game started on line %lu", game_start);
			} else {
				error("%s(%lu): spurious character '%c'",
				  asciiname, linenum, c);
				if (game_start)
					error("game started on line %lu", game_start);
			}
			return -1;
		}
	}

	if (!varlev) {
		/* end of main line */
		flush_eval();
		add_info_data();
		if (init_halfmove && to_colour(init_halfmove))
			move_count++;
		game->nmoves = (move_count + 1) / 2;
		if (init_comm) {
			add_comment(init_comm);
			free(init_comm);
			init_comm = NULL;
		}
	} else {
	/* end of variation */
		if (c != A_VAREND) {
			error("%s(%lu): end of game in middle of variation",
			  asciiname, linenum);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		if (!var_strip) {
			flush_eval();
			if (add_move(0x80) < 0)
				return -1;
		} else {
			eval = 0;
			poseval = 0;
			moveval = 0;
		}
	}

	return c ? 0 : 1;                  /* return 1 if EOF seen */
}

/*
 * ADD_MOVE
 *
 * Add a move to the movelist.
 */
static int
add_move(move)
	u_char move;
{
	if (nmoves >= MOVES_SZ) {
		error("%s(%lu): move buffer full!", asciiname, linenum);
		if (game_start)
			error("game started on line %lu", game_start);
		return -1;
	}
	*moves_ptr++ = move;
	nmoves++;
	return 0;
}

/*
 * GET_PGN_HEADER
 *
 * Extract data from a PGN header field.
 */
static int
get_pgn_header(token, s)
	int token;
	char *s;
{
	char *data, *cptr;
	int len, i;

	while (*s && *s != '"')
		s++;
	if (!*s)
		return -1;
	data = ++s;
	while (*s && *s != '"') {
		if (*s == '\\' && (*(s + 1) == '"' || *(s + 1) == '\\')) {
		/* process escaped quotes or backslashes */
			cptr = s;
			while (*cptr) {
				*cptr = *(cptr + 1);
				cptr++;
			}
		}
		s++;
	}

	if (*s != '"') {
		error("%s(%lu): unmatched quotes in PGN header", asciiname,
		  linenum);
		if (game_start)
			error("game started on line %lu", game_start);
		return -1;
	}
	*s = '\0';
	len = strlen(data);
	if (!len)
		return 0;
	if (strcmp(data, "?") == 0)
		return 0;

	switch (token) {
	case A_PGN_EVENT:
		event = xstrdup(data);
		if (!event)
			return -1;
		tidy_string(event);
		break;
	case A_PGN_SITE:
		site = xstrdup(data);
		if (!site)
			return -1;
		tidy_string(site);
		break;
	case A_PGN_DATE:
		game->year = atoi(data);
		break;
	case A_PGN_ROUND:
		round = xstrdup(data);
		if (!round)
			return -1;
		break;
	case A_PGN_WHITE:
		wplayer = xstrdup(data);
		if (!wplayer)
			return -1;
		tidy_name(wplayer);
		break;
	case A_PGN_BLACK:
		bplayer = xstrdup(data);
		if (!bplayer)
			return -1;
		tidy_name(bplayer);
		break;
	case A_PGN_RESULT:
		if (strcmp(data, "0-1") == 0)
			i = 0;
		else if (strcmp(data, "1/2-1/2") == 0)
			i = 1;
		else if (strcmp(data, "1-0") == 0)
			i = 2;
		else
			i = (4 << 2) | 3;
		set_result(game->header, i);
		break;
	case A_PGN_WHITEELO:
		game->w_elo = atoi(data);
		break;
	case A_PGN_BLACKELO:
		game->b_elo = atoi(data);
		break;
	case A_PGN_ECO:
		if (*data >= 'A' && *data <= 'E') {
			game->eco_letter = *data;
		} else if (*data >= 'a' && *data <= 'e') {
			game->eco_letter = *data + 'A' - 'a';
		} else {
			error("%s(%lu): invalid ECO code", asciiname, linenum);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		data++;
		if (isdigit(*data) && isdigit(*(data + 1))) {
			game->eco_main = atoi(data);
		} else {
			error("%s(%lu): invalid ECO code", asciiname, linenum);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		data += 2;
		if (*data == '/' || *data == '_') {
			data++;
			if (isdigit(*data) && isdigit(*(data + 1))) {
				game->eco_sub = atoi(data);
			} else {
				error("%s(%lu): invalid ECO code", asciiname, linenum);
				if (game_start)
					error("game started on line %lu", game_start);
				return -1;
			}
		}
		break;
	case A_PGN_ANNOTATOR:
		annotator = xstrdup(data);
		if (!annotator)
			return -1;
		tidy_name(annotator);
		break;
	case A_PGN_FEN:
		if (parse_fen(data) < 0)
			return -1;
		break;
	default:
		break;
	}
	return 0;
}

/*
 * TIDY_NAME
 *
 * Remove space after comma, remove trailing period.
 */
static void
tidy_name(s)
	char *s;
{
	int len;
	char *cptr;

 /* change all occurances of '-' to '_' */
	char_change(s, '-', '_');

 /* remove space after comma */
	cptr = index(s, ',');
	if (cptr) {
		if (*(cptr + 1) == ' ') {
			cptr++;
			len = strlen(cptr + 1);
			strncpy(cptr, cptr + 1, len);
			*(cptr + len) = '\0';
		}
	}
 /* remove space before comma */
	cptr = index(s, ',');
	if (cptr) {
		if (*(cptr - 1) == ' ') {
			cptr--;
			len = strlen(cptr + 1);
			strncpy(cptr, cptr + 1, len);
			*(cptr + len) = '\0';
		}
	}
 /* remove trailing period */
	cptr = s;
	len = strlen(cptr);
	if (len >= 3) {
		cptr += len - 1;
		if (*cptr == '.' && isupper(*(cptr - 1)) && !isalpha(*(cptr - 2)))
			*cptr = '\0';
	}

 /* generally tidy the string */
	tidy_string(s);
}

/*
 * ADD_INFO_DATA
 *
 * Add the captured game information to the game structure.
 */
static int
add_info_data()
{
	if (wplayer) {
		add_to_field(PLAYER_FIELD, wplayer, NULL, 0, 0);
		free(wplayer);
		wplayer = NULL;
	}
	if (bplayer) {
		add_to_field(PLAYER_FIELD, bplayer, "-", 0, 0);
		free(bplayer);
		bplayer = NULL;
	}
 /* check that "event" doesn't contain data from "site" */
	if (event && site) {
		if (strncmp(event, site, strlen(event)) == 0) {
			free(event);
			event = NULL;
		}
	}
	if (event) {
		add_to_field(SOURCE_FIELD, event, NULL, 0, 0);
		free(event);
		event = NULL;
	}
	if (site) {
		add_to_field(SOURCE_FIELD, site, ", ", 0, 0);
		free(site);
		site = NULL;
	}
	if (round) {
		add_to_field(SOURCE_FIELD, round, " ", '(', ')');
		free(round);
		round = NULL;
	}
	if (annotator) {
		add_to_field(SOURCE_FIELD, annotator, " ", '[', ']');
		free(annotator);
		annotator = NULL;
	}
	return 0;
}

/*
 * ADD_TO_FIELD
 *
 * Add data to the game fields in a controlled way.
 *
 * "type" is either "SOURCE_FIELD" or "PLAYER_FIELD",
 * "data" is the string to add,
 * "sep" is the string to add if data already exists in the field,
 * "leadchar" is the character to prefix the added data with,
 * "endchar" is the character to suffix the added data with.
 */
static int
add_to_field(type, data, sep, leadchar, endchar)
	int type;
	char *data, *sep, leadchar, endchar;

{
	u_char *field;
	int ret, len, datalen, seplen;

	ret = 0;
	if (type == SOURCE_FIELD) {
		field = game->sinfo;
		len = game->slen;
	} else if (type == PLAYER_FIELD) {
		field = game->pinfo;
		len = game->plen;
	} else {
		return -1;
	}

	field += len;

	if (len && sep) {
		seplen = strlen(sep);
		if (seplen + len > 47) {
			ret = -1;
			goto quit;
		}
		strcpy((char *) field, sep);
		field += seplen;
		len += seplen;
	}
	if (leadchar) {
		if (len + 1 > 47) {
			ret = -1;
			goto quit;
		}
		*field++ = leadchar;
		len++;
	}
	datalen = strlen(data);
	if (len + datalen > 47)
		datalen = 47 - len;
	if (datalen <= 0) {
		ret = -1;
		goto quit;
	}
	strncpy((char *) field, data, datalen);
	field += datalen;
	len += datalen;

	if (endchar) {
		if (len + 1 > 47) {
			ret = -1;
			goto quit;
		}
		*field++ = endchar;
		len++;
	}
quit:
	if (type == SOURCE_FIELD)
		game->slen = len;
	else if (type == PLAYER_FIELD)
		game->plen = len;
	return ret;
}

/*
 * PARSE_FEN
 *
 * Parse a Forsythe-Edwards Notation string.
 */
static int
parse_fen(s)
	char *s;
{
	int file, rank, colour, move, len;
	u_char piece, *board;
	char *args[6];

	for (file = 0; file < 6; file++) {
		args[file] = get_word(s, &len);
		if (!args[file]) {
			error("%s(%lu): not enough arguments in FEN string",
			  asciiname, linenum);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		s = args[file] + len;
		*s++ = '\0';
	}

	board = game->board;
	bzero((char *) board, 64);

	s = args[0];
	for (rank = 7; rank >= 0; rank--, s++) {
		for (file = 0; file < 8 && *s && *s != '/'; file++, s++) {
			if (isdigit(*s)) {
				file += (*s - '0') - 1;
				continue;
			} else if (isalpha(*s)) {
				piece = text2piece(*s);
				if (piece != 0xff) {
					board[to_offset(file, rank)] = piece;
					continue;
				}
			}
			error("%s(%lu): invalid piece list character ('%c') in FEN string",
			  asciiname, linenum, *s);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
	}
	copy_board(board, cb_board);

	s = args[1];
	if (*s == 'w') {
		colour = 0;
	} else if (*s == 'b') {
		colour = 8;
	} else {
		error("%s(%lu): invalid active colour ('%c') in FEN string",
		  asciiname, linenum, *s);
		if (game_start)
			error("game started on line %lu", game_start);
		return -1;
	}

	s = args[2];
	if (*s != '-') {
		while (*s) {
			switch (*s) {
			case 'K':
				set_wcs(game->header, 1);
				break;
			case 'Q':
				set_wcl(game->header, 1);
				break;
			case 'k':
				set_bcs(game->header, 1);
				break;
			case 'q':
				set_bcl(game->header, 1);
				break;
			default:
				error("%s(%lu): invalid castling availability ('%c') in FEN string",
				  asciiname, linenum, *s);
				if (game_start)
					error("game started on line %lu", game_start);
				return -1;
			}
			s++;
		}
	}
	s = args[3];
	if (*s != '-') {
		if (*s >= 'a' && *s <= 'h') {
			game->ep = *s - 'a' + 1;
			cb_enpassant = game->ep;
		} else {
			error("%s(%lu): invalid en-passant file ('%c') in FEN string",
			  asciiname, linenum, *s);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		s++;
		if ((colour && *s != '3') || (!colour && *s != '6')) {
			error("%s(%lu): invalid en-passant rank ('%c') in FEN string",
			  asciiname, linenum, *s);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
	}
	s = args[4];
	while (*s) {
		if (!isdigit(*s)) {
			error("%s(%lu): invalid halfmove clock number ('%c') if FEN string",
			  asciiname, linenum, *s);
			if (game_start)
				error("game started on line %lu", game_start);
			return -1;
		}
		s++;
	}

	s = args[5];
	move = atoi(s);
	if (!move) {
		error("%s(%lu): invalid fullmove number in FEN string",
		  asciiname, linenum);
		if (game_start)
			error("game started on line %lu", game_start);
		return -1;
	}
	init_halfmove = to_halfmove(move, colour);
	game->halfmove = init_halfmove;
	set_partial(game->header);

	return 0;
}

/*
 * DECODE_NAG
 *
 * Convert a Numeric Annotation Glyph into a value in "eval", "poseval" or
 * "moveval".
 */
static void
decode_nag(s)
	char *s;
{
	int i;
	struct nag *nag;

	i = atoi(s + 1);
	if (*s != '$' || i <= 0 || i > MAX_NAG) {
		error("%s(%lu): invalid NAG \"%s\"", asciiname, linenum,
		  s);
		if (game_start)
			error("game started on line %lu", game_start);
		return;
	}
	nag = &nags[i];
	switch (nag->var) {
	case VAR_EVAL:
		eval = nag->value;
		break;
	case VAR_POSEVAL:
		poseval = nag->value;
		break;
	case VAR_MOVEVAL:
		moveval = nag->value;
		break;
	default:
		break;
	}
}

/*
 * FLUSH_EVAL
 *
 * Add "eval", "poseval" or "moveval" are non-zero then append a comment
 * to store them, and set bit-7 of the last move.
 */
static int
flush_eval()
{
	if (eval || poseval || moveval) {
		if (comm_full)
			return -1;

		add_comm(eval);
		if (poseval || moveval) {
			add_comm(poseval);
			if (moveval)
				add_comm(moveval);
		}
		add_comm(0xff);
		*(moves_ptr - 1) |= 0x80;
		eval = 0;
		poseval = 0;
		moveval = 0;
	}
	return 0;
}

/*
 * ADD_COMMENT
 *
 * Tidy and insert comments into game data.  Only returns -1 if the comment
 * buffer is full when adding the comment header.
 *
 * Returns: 0 if the comment was added OK, -1 if the comment buffer is full,
 *          1 if the comment was discarded because it was "empty".
 */
static int
add_comment(s)
	char *s;
{
	int nrows, ncols, len, discard_text;
	char *word;

	if (comm_full)
		return -1;


    /* check if the textual part of the comment is non-whitespace, and if
       so discard it (later) */
    discard_text = 0;
    if (*s) {
        char *cptr = s;
        while (*cptr) {
            if (!isspace(*cptr)) {
                break;
            }
            cptr++;  
        }
        if (!*cptr) {
            /* we found only whitespace characters, so discard the comment */
            discard_text = 1;
            empty_comms_skipped++;
        }
    }

    /* if there is no textual comment and no evaluation type comment then
       return now as there is nothing to do */
    if (discard_text && !eval && !poseval && !moveval) {
        return 1;
    }

    /* store comment markers */
	add_comm((u_char)eval);
	add_comm((u_char)poseval);
	add_comm((u_char)moveval);
	add_comm(0);
	eval = 0;
	poseval = 0;
	moveval = 0;

    /* if there is no textual comment then return now after inserting a
       comment terminator */
    if (discard_text) {
        add_comm(0xff);
        return 0;
    }   

    /* if doing new-style comments then simply throw them in, else format the
	   comment to 3 rows of 78 characters... */
	if (long_comments) {
		while (*s) {
			if (add_comm(*s++) < 0) {
				return 0;
			}
		}
	} else {
		for (nrows = 0, ncols = 0; nrows < 3; nrows++) {
			while (ncols < 79) {
				word = get_word(s, &len);
				if (!word)
					break;
				if (len > 78) {
					len = (78 - ncols) - (ncols ? 1 : 0);
				} else if (ncols + len + (ncols ? 1 : 0) > 78) {
					if (nrows < 2) {       /* first 2 lines */
						break;
					} else {               /* last line */
						len = (78 - ncols) - (ncols ? 1 : 0);
					}
				}
				if (len < 0) {
					len = 0;
					break;
				}
				if (ncols) {
					if (add_comm(' ') < 0)
						return 0;
					ncols++;
				}
				while (len--) {
					if (add_comm(*word++) < 0)
						return 0;
					ncols++;
				}
				s = word;
			}
  		    /* add a newline if first 2 lines and more text to follow */
			if (nrows < 2 && get_word(s, NULL)) {
				if (add_comm(0xfe) < 0)
					return 0;
				ncols = 0;
			}
		}
	}
	add_comm(0xff);
	return 0;
}

/*
 * ADD_COMM
 *
 * Actually put a comment byte into the buffer.
 */
static int
add_comm(comm)
	u_char comm;
{
	if (ncomms >= COMMS_SZ) {
		error("%s(%lu): comment buffer full!", asciiname, linenum);
		if (game_start)
			error("game started on line %lu", game_start);
		comm_full++;
		ncomms = COMMS_SZ;
		comms_ptr = comms + (COMMS_SZ - 1);
		*comms_ptr = 0xff;             /* terminate last comment */
		return -1;
	}
	*comms_ptr++ = comm;
	ncomms++;
	return 0;
}
