
// _________________________________________________________________
//
//                           SBS CabLib.c
//             Windows Cabinet Management Library V1.00
//                   01-20-1997 Sven B. Schreiber
//                          sbs@orgon.com
// _________________________________________________________________

#include <windows.h>
#include <fcntl.h>
#include <stat.h>
#include <fdi.h>
#include "CabLib.h"

// =================================================================
// REVISION HISTORY
// =================================================================

/*

01-20-1997  V1.00  SBS  Original version.

*/

// =================================================================
// GLOBAL VARIABLES
// =================================================================

HINSTANCE hInst = NULL;
HFDI      hfdi  = NULL;
ERF       erf;

// =================================================================
// GLOBAL STRINGS
// =================================================================

BYTE abMainError     [] = "CabLib Error";

BYTE abTitleSaveFile [] = "Save File";
BYTE abNone          [] = "<none>";
BYTE abNull          [] = "";

BYTE abErrorCreate   [] = "Unable to create file";
BYTE abErrorOpen     [] = "Unable to open file";
BYTE abErrorClose    [] = "Unable to close file";
BYTE abErrorExtract  [] = "Unable to extract file";

// =================================================================
// AUXILIARY ROUTINES
// =================================================================

PBYTE WINAPI GetFileName (PBYTE pbPath)
    {
    DWORD i, j;

    j = i = (pbPath [0] ? (pbPath [1] == ':' ? 2 : 0) : 0);

    while (pbPath [i++])
        {
        if (pbPath [i-1] == '\\') j = i;
        }
    return pbPath+j;
    }

// -----------------------------------------------------------------

PBYTE WINAPI RemoveFileName (PBYTE pbPath)
    {
    PBYTE pbFileName;
    DWORD i;

    i = (pbPath [0] ? (pbPath [1] == ':' ? 2 : 0) : 0);
    pbFileName = GetFileName (pbPath+i);

    if ((pbFileName != (pbPath+i)) && (pbFileName [-1] == '\\'))
        {
        pbFileName--;
        }
    if ((pbFileName == (pbPath+i)) && (pbFileName [0] == '\\'))
        {
        pbFileName++;
        }
    pbFileName [0] = 0;
    return pbPath;
    }

// -----------------------------------------------------------------

BOOL WINAPI GetDefaultDestination (PBYTE pbProfile,
                                   PBYTE pbSection,
                                   PBYTE pbKey,
                                   PBYTE pbPath,
                                   DWORD dBytes)
    {
    DWORD dSize;

    if (dBytes || (dSize = 0))
        {
        dSize = GetPrivateProfileString (pbSection,
                                         pbKey,
                                         abNull,
                                         pbPath,
                                         dBytes,
                                         pbProfile);
        }
    return (dSize != 0);
    }

// -----------------------------------------------------------------

BOOL WINAPI SetDefaultDestination (PBYTE pbProfile,
                                   PBYTE pbSection,
                                   PBYTE pbKey,
                                   PBYTE pbPath)
    {
    return WritePrivateProfileString (pbSection,
                                      pbKey,
                                      pbPath,
                                      pbProfile);
    }

// -----------------------------------------------------------------

BOOL WINAPI NormalizePath (PBYTE pbPath,
                           DWORD dBytes)
    {
    DWORD n;
    BOOL  fOK = TRUE;

    if ((n = lstrlen (pbPath)) &&
        (pbPath [n-1] != '\\') &&
        (fOK = ((n+1) < dBytes)))
        {
        pbPath [n]   = '\\';
        pbPath [n+1] = 0;
        }
    return fOK;
    }

// -----------------------------------------------------------------

int printfMessage (HWND  hWnd,
                   UINT  uType,
                   PBYTE pbCaption,
                   PBYTE pbFormat,
                   ...)
    {
    BYTE abBuffer [MAX_MESSAGE];

    wvsprintf (abBuffer, pbFormat, (PVOID) ((&pbFormat)+1));
    return MessageBox (hWnd, abBuffer, pbCaption, uType);
    }

// =================================================================
// FDI CALLBACKS
// =================================================================

PVOID DIAMONDAPI FDIAlloc (ULONG cb)
    {
    return LocalAlloc (LMEM_FIXED | LMEM_ZEROINIT, cb);
    }

// -----------------------------------------------------------------

void DIAMONDAPI FDIFree (PVOID pv)
    {
    if (pv != NULL) LocalFree (pv);
    return;
    }

// -----------------------------------------------------------------

int DIAMONDAPI FDIOpen (PSTR pszFile,
                        int  oflag,
                        int  pmode)
    {
    HANDLE hFile;
    DWORD  dRwMask  = (_O_RDONLY | _O_WRONLY | _O_RDWR);
    DWORD  dAccess  = 0;
    DWORD  dOptions = 0;
    DWORD  dFlags   = 0;

    if ((oflag & dRwMask) == _O_RDONLY)
        {
        dAccess |= GENERIC_READ;
        }
    if ((oflag & dRwMask) == _O_WRONLY)
        {
        dAccess |= GENERIC_WRITE;
        }
    if ((oflag & dRwMask) == _O_RDWR)
        {
        dAccess |= (GENERIC_READ | GENERIC_WRITE);
        }
    if (oflag & _O_CREAT)
        {
        dOptions |= (oflag & _O_EXCL ? CREATE_NEW : OPEN_ALWAYS);

        if (oflag & _O_SHORT_LIVED)
            {
            dFlags |= FILE_ATTRIBUTE_TEMPORARY;
            }
        if (oflag & _O_TEMPORARY)
            {
            dFlags |= (FILE_ATTRIBUTE_TEMPORARY |
                       FILE_FLAG_DELETE_ON_CLOSE);
            }
        if (pmode & _S_IWRITE)
            {
            dFlags |= FILE_ATTRIBUTE_NORMAL;
            }
        else
            {
            dFlags |= FILE_ATTRIBUTE_READONLY;
            }
        }
    else
        {
        dOptions |= OPEN_EXISTING;
        }
    if (oflag & _O_RANDOM)
        {
        dFlags |= FILE_FLAG_RANDOM_ACCESS;
        }
    if (oflag & _O_SEQUENTIAL)
        {
        dFlags |= FILE_FLAG_SEQUENTIAL_SCAN;
        }
    hFile = CreateFile (pszFile,
                        dAccess,
                        FILE_SHARE_READ,
                        NULL,
                        dOptions,
                        dFlags,
                        NULL);

    if ((hFile != INVALID_HANDLE_VALUE) && (oflag & _O_TRUNC))
        {
        if (!SetEndOfFile (hFile))
            {
            CloseHandle (hFile);
            hFile = INVALID_HANDLE_VALUE;
            }
        }
    if ((hFile != INVALID_HANDLE_VALUE) && (oflag & _O_APPEND))
        {
        if (SetFilePointer (hFile, 0, NULL, FILE_END) == MAXDWORD)
            {
            CloseHandle (hFile);
            hFile = INVALID_HANDLE_VALUE;
            }
        }
    return (hFile != INVALID_HANDLE_VALUE ? (int) hFile : -1);
    }

// -----------------------------------------------------------------

UINT DIAMONDAPI FDIRead (int   hf,
                         PVOID pv,
                         UINT  cb)
    {
    DWORD n;

    return (hf == -1
            ? (UINT) -1
            : (!cb
               ? 0
               : (ReadFile ((HANDLE) hf, pv, cb, &n, NULL)
                  ? n
                  : (UINT) -1)));
    }

// -----------------------------------------------------------------

UINT DIAMONDAPI FDIWrite (int   hf,
                          PVOID pv,
                          UINT  cb)
    {
    DWORD n;

    return (hf == -1
            ? (UINT) -1
            : (!cb
               ? 0
               : (WriteFile ((HANDLE) hf, pv, cb, &n, NULL)
                  ? n
                  : (UINT) -1)));
    }

// -----------------------------------------------------------------

int DIAMONDAPI FDIClose (int hf)
    {
    return (hf == -1
            ? -1
            : (CloseHandle ((HANDLE) hf) ? 0 : -1));
    }

// -----------------------------------------------------------------

long DIAMONDAPI FDISeek (int  hf,
                         long dist,
                         int  seektype)
    {
    DWORD dMethod = 0;
    BOOL  fOK     = (hf != -1);

    switch (seektype)
        {
        case SEEK_SET: dMethod = FILE_BEGIN;   break;
        case SEEK_CUR: dMethod = FILE_CURRENT; break;
        case SEEK_END: dMethod = FILE_END;     break;
        default:       fOK     = FALSE;        break;
        }
    return (!fOK
            ? -1
            : SetFilePointer ((HANDLE) hf, dist, NULL, dMethod));
    }

// -----------------------------------------------------------------

int DIAMONDAPI FDIExtract (FDINOTIFICATIONTYPE fdint,
                           PFDINOTIFICATION    pfdin)
    {
    FILETIME ft, lft;
    BYTE     abFile [MAX_PATH];
    int      hf, iResult;

    switch (fdint)
        {
        case fdintCOPY_FILE:

            if (lstrcmpi (pfdin->psz1,
                          ((PEXTRACT_FILE) pfdin->pv)->pbEntry))
                {
                iResult = CAB_NOTIFY_SKIP;
                }
            else
                {
                hf = FDIOpen (((PEXTRACT_FILE) pfdin->pv)->pbFile,
                              _O_RDWR | _O_CREAT | _O_TRUNC,
                              _S_IREAD | _S_IWRITE);
                if (hf != -1)
                    {
                    iResult = hf;
                    }
                else
                    {
                    printfMessage
                        (((PEXTRACT_FILE) pfdin->pv)->hWnd,
                          MB_ICONERROR | MB_OK,
                          abMainError,
                          "%s \"%s\"",
                          abErrorCreate,
                          ((PEXTRACT_FILE) pfdin->pv)->pbFile);

                    iResult = CAB_NOTIFY_ABORT;
                    }
                }
            break;

        case fdintNEXT_CABINET:

            if (lstrlen (pfdin->psz3) + lstrlen (pfdin->psz1)
                < MAX_PATH)
                {
                lstrcpy (abFile, pfdin->psz3);
                lstrcat (abFile, pfdin->psz1);
                hf = FDIOpen (abFile, _O_RDONLY, 0);
                }
            else
                {
                hf = -1;
                }
            if (hf != -1)
                {
                FDIClose (hf);
                iResult = CAB_NOTIFY_OK;
                }
            else
                {
                printfMessage (((PEXTRACT_FILE) pfdin->pv)->hWnd,
                               MB_ICONERROR | MB_OK,
                               abMainError,
                               "%s \"%s%s\"",
                               abErrorOpen,
                               pfdin->psz3,
                               pfdin->psz1);

                iResult = CAB_NOTIFY_ABORT;
                }
            break;

        case fdintCLOSE_FILE_INFO:

            DosDateTimeToFileTime (pfdin->date, pfdin->time, &lft);
            LocalFileTimeToFileTime (&lft, &ft);
            SetFileTime ((HANDLE) pfdin->hf, &ft, &ft, &ft);

            if (FDIClose (pfdin->hf) == -1)
                {
                printfMessage (((PEXTRACT_FILE) pfdin->pv)->hWnd,
                               MB_ICONERROR | MB_OK,
                               abMainError,
                               "%s \"%s\"",
                               abErrorClose,
                               ((PEXTRACT_FILE) pfdin->pv)->pbFile);
                }
            iResult = CAB_NOTIFY_OK;
            break;

        default:

            iResult = CAB_NOTIFY_OK;
            break;
        }
    return iResult;
    }

// =================================================================
// CABINET MANAGEMENT
// =================================================================

int WINAPI LoadCabinetHeader (PBYTE       pbCabinet,
                              PCAB_HEADER pch)
    {
    int hf;

    if ((hf = FDIOpen (pbCabinet, _O_RDONLY, 0)) != -1)
        {
        if ((FDIRead (hf, pch, CAB_HEADER_) != CAB_HEADER_  ) ||
            (pch->sig                       != CAB_SIGNATURE) ||
            (pch->version                   != CAB_VERSION  ) ||
            (pch->cFolders                  == 0            ) ||
            (pch->cFiles                    == 0            ) ||
            (pch->coffFiles                 <= CAB_HEADER_  ))
            {
            FDIClose (hf);
            hf = -1;
            }
        }
    return hf;
    }

// -----------------------------------------------------------------

BOOL WINAPI TestCabinetHeader (PBYTE pbCabinet)
    {
    int        hf;
    CAB_HEADER ch;

    if ((hf = LoadCabinetHeader (pbCabinet, &ch)) != -1)
        {
        FDIClose (hf);
        }
    return (hf != -1);
    }

// -----------------------------------------------------------------

BOOL WINAPI ReadCabinetBlock (int   hf,
                              PBYTE pbData,
                              DWORD dOffset,
                              DWORD dTotal)
    {
    return ((dOffset <= dTotal) &&
            (FDIRead (hf, pbData+dOffset, dTotal-dOffset)
             == dTotal-dOffset));
    }

// -----------------------------------------------------------------

DWORD WINAPI CloseCabinet (HCABINET hc)
    {
    DWORD dOffset = 0;

    if (hc != NULL)
        {
        dOffset = hc->dOffset;
        if (hc->hf     != -1)   FDIClose  (hc->hf);
        if (hc->pbData != NULL) LocalFree (hc->pbData);
        LocalFree (hc);
        }
    return dOffset;
    }

// -----------------------------------------------------------------

HCABINET WINAPI OpenCabinet (PBYTE pbCabinet)
    {
    HCABINET hc;
    PBYTE    pbData;
    DWORD    i;
    BOOL     fOK = FALSE;

    hc = LocalAlloc (LMEM_FIXED, CAB_CONTROL_);

    if (hc != NULL)
        {
        hc->hf = -1;
        pbData = LocalAlloc (LMEM_FIXED, CAB_HEADER_);

        if (pbData != NULL)
            {
            hc->hf = LoadCabinetHeader (pbCabinet,
                                        (PCAB_HEADER) pbData);

            if (hc->hf != -1)
                {
                hc->dFileList = ((PCAB_HEADER) pbData)->coffFiles;

                pbData  = LocalReAlloc (pbData,
                                        hc->dFileList,
                                        LMEM_MOVEABLE);

                if ((pbData != NULL) &&
                    ReadCabinetBlock (hc->hf,
                                      pbData,
                                      CAB_HEADER_,
                                      hc->dFileList))
                    {
                    hc->dExtraData = i = CAB_HEADER_;

                    if (((PCAB_HEADER) pbData)->flags &
                        CAB_FLAG_RESERVE)
                        {
                        hc->dReserve  = *((PDWORD) (pbData+i));
                        i            += sizeof (DWORD);

                        hc->pReserve  = (PVOID) (pbData+i);
                        i            += hc->dReserve;
                        }
                    else
                        {
                        hc->dReserve = 0;
                        hc->pReserve = NULL;
                        }
                    hc->dCabList = i;

                    if (((PCAB_HEADER) pbData)->flags &
                        CAB_FLAG_HASPREV)
                        {
                        hc->pbPrevCab  = pbData+i;
                        while ((i < hc->dFileList) &&
                               (pbData [i++]));

                        hc->pbPrevDisk = pbData+i;
                        while ((i < hc->dFileList) &&
                               (pbData [i++]));
                        }
                    else
                        {
                        hc->pbPrevCab  = abNone;
                        hc->pbPrevDisk = abNull;
                        }
                    if (((PCAB_HEADER) pbData)->flags &
                        CAB_FLAG_HASNEXT)
                        {
                        hc->pbNextCab  = pbData+i;
                        while ((i < hc->dFileList) &&
                               (pbData [i++]));

                        hc->pbNextDisk = pbData+i;
                        while ((i < hc->dFileList) &&
                               (pbData [i++]));
                        }
                    else
                        {
                        hc->pbNextCab  = abNone;
                        hc->pbNextDisk = abNull;
                        }
                    if (i+CAB_FOLDER_ <= hc->dFileList)
                        {
                        hc->dFolderList = i;
                        hc->dFileData   = ((PCAB_FOLDER) (pbData+i))
                                          ->coffCabStart;

                        pbData  = LocalReAlloc (pbData,
                                                hc->dFileData,
                                                LMEM_MOVEABLE);

                        if ((pbData != NULL) &&
                            ReadCabinetBlock (hc->hf,
                                              pbData,
                                              hc->dFileList,
                                              hc->dFileData))
                            {
                            fOK = TRUE;
                            }
                        }
                    }
                }
            }
        if (fOK)
            {
            hc->dSetId     = ((PCAB_HEADER) pbData)->setID;
            hc->dCabNumber = ((PCAB_HEADER) pbData)->iCabinet;
            hc->dFolders   = ((PCAB_HEADER) pbData)->cFolders;
            hc->dFiles     = ((PCAB_HEADER) pbData)->cFiles;
            hc->dCabSize   = ((PCAB_HEADER) pbData)->cbCabinet;
            hc->dOffset    = hc->dFileList;
            hc->dSize      = hc->dFileData;
            hc->pbData     = pbData;
            }
        else
            {
            CloseCabinet (hc);
            hc = NULL;
            }
        }
    return hc;
    }

// -----------------------------------------------------------------

BOOL WINAPI ScanCabinet (HCABINET   hc,
                         PCAB_ENTRY pce,
                         PBYTE      pbEntry,
                         DWORD      dBytes)
    {
    DWORD i;
    BOOL  fOK = FALSE;

    if (hc->dFiles)
        {
        i = hc->dOffset + CAB_ENTRY_;
        while ((i < hc->dSize) && hc->pbData [i]) i++;
        if (i < hc->dSize)
            {
            if (dBytes)
                {
                lstrcpyn (pbEntry,
                          hc->pbData + hc->dOffset + CAB_ENTRY_,
                          dBytes);
                }
            CopyMemory (pce, hc->pbData + hc->dOffset, CAB_ENTRY_);
            hc->dFiles--;
            hc->dOffset = i+1;
            fOK = TRUE;
            }
        }
    return fOK;
    }

// -----------------------------------------------------------------

BOOL WINAPI FindCabinetEntry (PBYTE      pbCabinet,
                              PBYTE      pbEntry,
                              DWORD      dOffset,
                              PCAB_ENTRY pce)
    {
    PBYTE pbData;
    int   hf;
    DWORD i, n;
    BOOL  fOK = FALSE;

    if (dOffset && (dOffset != MAXDWORD))
        {
        if ((pbData = LocalAlloc (LMEM_FIXED, dOffset)) != NULL)
            {
            if ((hf = FDIOpen (pbCabinet, _O_RDONLY, 0)) != -1)
                {
                if (FDIRead (hf, pbData, dOffset) == dOffset)
                    {
                    i = ((PCAB_HEADER) pbData)->coffFiles;
                    n = ((PCAB_HEADER) pbData)->cFiles;

                    while ((!fOK) && (n--))
                        {
                        if (!lstrcmp (pbData+i+CAB_ENTRY_, pbEntry))
                            {
                            CopyMemory (pce, pbData+i, CAB_ENTRY_);
                            fOK = TRUE;
                            }
                        else
                            {
                            i += (CAB_ENTRY_ +
                                  lstrlen (pbData+i+CAB_ENTRY_) +
                                  1);
                            }
                        }
                    }
                FDIClose (hf);
                }
            LocalFree (pbData);
            }
        }
    return fOK;
    }

// -----------------------------------------------------------------

BOOL WINAPI ExtractFileFromCabinet (HWND  hWnd,
                                    PBYTE pbCabinet,
                                    PBYTE pbEntry,
                                    PBYTE pbProfile,
                                    PBYTE pbSection,
                                    PBYTE pbKey)
    {
    OPENFILENAME ofn;
    EXTRACT_FILE ef;
    HCURSOR      hCursor;
    BYTE         abDirectory [MAX_PATH];
    BYTE         abPath      [MAX_PATH];
    BYTE         abFile      [MAX_PATH];
    BOOL         fOK = FALSE;

    GetCurrentDirectory (MAX_PATH, abDirectory);

    GetDefaultDestination (pbProfile,
                           pbSection,
                           pbKey,
                           abPath,
                           MAX_PATH);

    lstrcpyn (abFile, pbEntry, MAX_PATH);

    ofn.lStructSize       = sizeof (OPENFILENAME);
    ofn.hInstance         = hInst;
    ofn.hwndOwner         = hWnd;
    ofn.lpstrFilter       = "Unspecified\0*\0";
    ofn.lpstrCustomFilter = NULL;
    ofn.nMaxCustFilter    = 0;
    ofn.nFilterIndex      = 1;
    ofn.lpstrFile         = abFile;
    ofn.nMaxFile          = MAX_PATH;
    ofn.lpstrFileTitle    = NULL;
    ofn.nMaxFileTitle     = 0;
    ofn.lpstrInitialDir   = (abPath [0] ? abPath : NULL);
    ofn.lpstrTitle        = abTitleSaveFile;
    ofn.nFileOffset       = 0;
    ofn.nFileExtension    = 0;
    ofn.lpstrDefExt       = NULL;
    ofn.lCustData         = (LPARAM) NULL;
    ofn.lpfnHook          = NULL;
    ofn.lpTemplateName    = NULL;
    ofn.Flags             = OFN_OVERWRITEPROMPT |
                            OFN_HIDEREADONLY    |
                            OFN_EXPLORER;

    fOK = GetSaveFileName (&ofn);
    SetCurrentDirectory (abDirectory);

    if (fOK)
        {
        if (NormalizePath (abDirectory, MAX_PATH))
            {
            hCursor = SetCursor (LoadCursor (NULL, IDC_WAIT));
            UpdateWindow (hWnd);

            ef.hWnd    = hWnd;
            ef.pbEntry = pbEntry;
            ef.pbFile  = abFile;

            fOK = FDICopy (hfdi,
                           pbCabinet,
                           abDirectory,
                           0,
                           FDIExtract,
                           NULL,
                           &ef);

            SetCursor (hCursor);

            if (!fOK)
                {
                printfMessage (hWnd,
                               MB_ICONERROR | MB_OK,
                               abMainError,
                               "%s \"%s\"",
                               abErrorExtract,
                               pbEntry);

                DeleteFile (abFile);
                }
            }
        SetDefaultDestination (pbProfile,
                               pbSection,
                               pbKey,
                               RemoveFileName (abFile));
        }
    return fOK;
    }

// =================================================================
// DLL ENTRY POINT
// =================================================================

BOOL WINAPI DllMain (HINSTANCE hInstance,
                     DWORD     dReason,
                     PVOID     pReserved)
    {
    switch (dReason)
        {
        case DLL_PROCESS_ATTACH:

            hInst = hInstance;

            hfdi = FDICreate (FDIAlloc,
                              FDIFree,
                              FDIOpen,
                              FDIRead,
                              FDIWrite,
                              FDIClose,
                              FDISeek,
                              cpuUNKNOWN,
                              &erf);

            return (hfdi != NULL);

        case DLL_PROCESS_DETACH:

            if (hfdi != NULL)
                {
                FDIDestroy (hfdi);
                hfdi = NULL;
                }
            return TRUE;

        default:

            return TRUE;
        }
    }

// =================================================================
// END OF PROGRAM
// =================================================================
